import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import PageTitle from '../../components/PageTitle';
import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import { fetchAgents } from '../../api/agent';
import LocalDate from '../../components/LocalDate';
import TabsWithUrl from '../../components/TabsWithUrl';
import { TabPane } from '../../components/Tabs';
import { getUrlHash, setUrlHash } from '../../utils/url';
import FetchTargetsView from '../shards/ShardStatusPage/FetchTargetsView';
import FetchingGraphsView from '../shards/ShardStatusPage/FetchingGraphsView';
import { isPreOrProdIntEnv } from '../../utils/env';

class AgentsPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      agents: [],
      duration: '1h',
      viewNewVersion: isPreOrProdIntEnv(),
    };
  }

  componentDidMount() {
    const provider = this.props.match.params.serviceProviderId;
    fetchAgents({ provider, pageSize: 'all' })
      .then((response) => {
        this.setState({ agents: response.result });
      });
  }

  onToggleViewNew = () => {
    this.setState({
      viewNewVersion: !this.state.viewNewVersion,
    });
  }

  onDurationChange = (duration) => {
    this.setState({ duration });
  };

  render() {
    const provider = this.props.match.params.serviceProviderId;
    const {
      duration,
      viewNewVersion,
    } = this.state;

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Agents of ${provider}`} />
        <TabsWithUrl activeTabId={getUrlHash()} setActiveTab={setUrlHash}>
          <TabPane label="Pull status">
            <FetchTargetsView
              projectId=""
              shardId=""
              serviceProviderId={provider}
              shardHosts={[]}
              search={this.props.location.search}
              history={this.props.history}
            />
          </TabPane>
          <TabPane label="Agents">
            <table className="table table-condensed table-hover">
              <thead>
                <tr>
                  <th>#</th>
                  <th>Hostname</th>
                  <th>Cluster id</th>
                  <th>Data port</th>
                  <th>Management port</th>
                  <th>Description</th>
                  <th>Last seen</th>
                  <th>Pull interval</th>
                  <th>Version</th>
                </tr>
              </thead>
              <tbody>
                {this.state.agents.map((agent, i) => (
                  <tr key={agent.id}>
                    <td>{i + 1}</td>
                    <td>{agent.hostname}</td>
                    <td>{agent.clusterId}</td>
                    <td>{agent.dataPort}</td>
                    <td>{agent.managementPort}</td>
                    <td>{agent.description}</td>
                    <td><LocalDate date={agent.lastSeen} /></td>
                    <td>{`${agent.pullIntervalSeconds}s`}</td>
                    <td>{agent.version}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </TabPane>
          <TabPane label="Fetching">
            <FetchingGraphsView
              projectId={provider}
              shardId={`${provider}_provider_${provider}_agent`}
              duration={duration}
              viewNewVersion={viewNewVersion}
              onToggleViewNew={this.onToggleViewNew}
              onDurationChange={this.onDurationChange}
            />
          </TabPane>
        </TabsWithUrl>
      </div>
    );
  }
}

AgentsPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
};

export default AgentsPage;
