/* eslint-disable max-classes-per-file */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

function listToStr(list) {
  return list ? list.join(', ') : '';
}

function strToList(str) {
  return str ? str.split(/\s*,\s*/) : [];
}

function wrapHandler(index, handler) {
  if (!handler) {
    return null;
  }
  return (event) => {
    event.preventDefault();
    event.stopPropagation();
    handler(index);
  };
}

export class ReadonlyRow extends PureComponent {
  constructor(props) {
    super(props);
    this.onClick = wrapHandler(props.index, props.onClick);
    this.onDelete = wrapHandler(props.index, props.onDelete);
  }

  render() {
    const { index, value } = this.props;
    return (
      <tr onClick={this.onClick}>
        <td>{index + 1}</td>
        <td>{value.label || ''}</td>
        <td>{listToStr(value.services)}</td>
        <td>{listToStr(value.types)}</td>
        <td>{value.crossFolder ? 'Yes' : 'No'}</td>
        {this.onDelete
        && (
        <td className="actions">
          <button type="button" onClick={this.onDelete}>
            <i className="glyphicon glyphicon-remove" />
          </button>
        </td>
        )}
      </tr>
    );
  }
}

ReadonlyRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: PropTypes.object.isRequired,
  onClick: PropTypes.func,
  onDelete: PropTypes.func,
};

ReadonlyRow.defaultProps = {
  onClick: null,
  onDelete: null,
};

export class EditableRow extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      label: props.value.label || '',
      services: listToStr(props.value.services),
      types: listToStr(props.value.types),
      crossFolder: !!props.value.crossFolder,
    };
  }

  componentDidMount() {
    this.labelInput.focus();
  }

  onOkClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.doComplete();
  };

  onKeyDown = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      event.stopPropagation();
      this.doComplete();
    } else if (event.which === 27) {
      event.preventDefault();
      this.props.onCancel();
    }
  };

  onInputChange = (event) => {
    const {
      type, name, value, checked,
    } = event.target;
    if (type === 'text') {
      this.setState({ [name]: value });
    } else if (type === 'checkbox') {
      this.setState({ [name]: checked });
    }
  };

  doComplete() {
    const { onOk, index } = this.props;
    const {
      label, services, types, crossFolder,
    } = this.state;
    const reference = {
      label,
      services: strToList(services),
      types: strToList(types),
      crossFolder,
    };
    onOk(index, reference);
  }

  render() {
    const { index } = this.props;
    const {
      label, services, types, crossFolder,
    } = this.state;
    return (
      <tr onKeyDown={this.onKeyDown}>
        <td>{index >= 0 ? index + 1 : ''}</td>
        <td>
          <input
            type="text" name="label" className="form-control"
            value={label} onChange={this.onInputChange}
            ref={(input) => { this.labelInput = input; }}
          />
        </td>
        <td>
          <input
            type="text" name="services" className="form-control"
            value={services} onChange={this.onInputChange}
          />
        </td>
        <td>
          <input
            type="text" name="types" className="form-control"
            value={types} onChange={this.onInputChange}
          />
        </td>
        <td>
          <input
            type="checkbox" name="crossFolder"
            checked={crossFolder}
            onChange={this.onInputChange}
          />
        </td>
        <td className="actions">
          <button type="button" onClick={this.onOkClick}>
            <i className="glyphicon glyphicon-ok" />
          </button>
        </td>
      </tr>
    );
  }
}

EditableRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: PropTypes.object,
  onOk: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

EditableRow.defaultProps = {
  value: {
    label: '', services: [], types: [], crossFolder: false,
  },
};
