import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import ServiceDashboardForm from './ServiceDashboardForm';

import {
  loadServiceDashboard,
  saveServiceDashboard,
  clearServiceDashboard,
  validateServiceDashboard,
} from '../../store/reducers/serviceProviders/serviceDashboard';
import WarningAlert from '../projects/alerts/WarningAlert';
import DashboardValidationResultModal from './DashboardValidationResultModal';
import LabeledValue from '../../components/LabeledValue';

class ServiceDashboardEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { isOpenValidation: false, value: undefined };
  }

  componentDidMount() {
    const id = this.props.match.params.serviceDashboardId || '';

    const { serviceDashboardData } = this.props;
    if (id) {
      const loadedId = serviceDashboardData.data ? serviceDashboardData.data.id : '';
      if (loadedId !== id) {
        this.props.loadServiceDashboard(id);
      }
    } else {
      this.props.clearServiceDashboard();
    }
  }

  onSubmit = (value) => {
    const { serviceProviderId } = this.props.match.params;
    const id = this.props.match.params.serviceDashboardId || '';
    const isNew = !id;

    this.props.validateServiceDashboard({ ...value, service: serviceProviderId })
      .then(() => {
        const { validationResultsData } = this.props;
        if (validationResultsData.validationIssues.length === 0) {
          const { history } = this.props;
          this.props.saveServiceDashboard({ ...value, service: serviceProviderId }, isNew)
            .then(() => history.push(`/admin/serviceProviders/${serviceProviderId}/serviceDashboards/${value.serviceDashboardId}`));
        } else {
          this.setState({ isOpenValidation: true, value });
        }
      });
  };

  handleHideValidationModal = () => {
    this.setState({ isOpenValidation: false, value: undefined });
  };

  handleConfirmValidationModal = () => {
    const { history } = this.props;
    const { value } = this.state;
    const { serviceProviderId } = this.props.match.params;
    const id = this.props.match.params.serviceDashboardId || '';
    const isNew = !id;
    this.props.saveServiceDashboard({ ...value, service: serviceProviderId }, isNew)
      .then(() => history.push(`/admin/serviceProviders/${serviceProviderId}/serviceDashboards/${value.serviceDashboardId}`));
  };

  render() {
    const id = this.props.match.params.serviceDashboardId || '';
    const isNewServiceDashboard = !id;
    const data = this.props.serviceDashboardData || {};
    const title = isNewServiceDashboard ? 'New service dashboard' : `Service dashboard ${id}`;

    const validateButton = (
      <>
        <DashboardValidationResultModal
          isOpenValidation={this.state.isOpenValidation}
          onConfirm={this.handleConfirmValidationModal}
          onCancel={this.handleHideValidationModal}
        />
      </>
    );
    const validateField = (
      <LabeledValue label="" value={validateButton} />
    );

    let content = null;
    if (data.loading) {
      content = <div>Loading...</div>;
    } else if (data.error) {
      content = <WarningAlert title="Error!" message={data.error} />;
    } else if (isNewServiceDashboard || data.data) {
      const serviceDashboard = data.data || {};
      content = (
        <ServiceDashboardForm
          serviceDashboard={serviceDashboard}
          isNew={isNewServiceDashboard}
          onSubmit={this.onSubmit}
        />
      );
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {content}
        {validateField}
      </div>
    );
  }
}

ServiceDashboardEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  validationResultsData: PropTypes.object,
  serviceDashboardData: PropTypes.object.isRequired,
  loadServiceDashboard: PropTypes.func.isRequired,
  saveServiceDashboard: PropTypes.func.isRequired,
  clearServiceDashboard: PropTypes.func.isRequired,
  validateServiceDashboard: PropTypes.func.isRequired,
};

ServiceDashboardEditPage.defaultProps = {
  validationResultsData: {
    validationIssues: [],
  },
};

const mapStateToProps = (state) => ({
  serviceDashboardData: state.serviceDashboard,
  validationResultsData: state.serviceDashboardValidationResult,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadServiceDashboard,
  saveServiceDashboard,
  validateServiceDashboard,
  clearServiceDashboard,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(ServiceDashboardEditPage);
