import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormButtons from '../../components/forms/FormButtons';
import FormInput from '../../components/forms/FormInput';
import { inputTargetValue } from '../../utils/inputs';
import FormArea from '../../components/forms/FormArea';
import ProtoTextEditor from '../../components/ProtoTextEditor';
import FormElement from '../../components/forms/FormElement';
import TableEditor from '../../components/TableEditor';
import * as ServiceDashboardOwnerTable from './ServiceDashboardOwnerTable';

function tryParse(value, defaultValue) {
  try {
    return JSON.parse(value) || defaultValue;
  } catch (e) {
    console.error(e);
    return defaultValue;
  }
}

class ServiceDashboardForm extends PureComponent {
  static serviceDashboardToState = (serviceDashboard, isNew) => {
    if (isNew) {
      return {
        serviceDashboardId: '',
        description: '',
        widgets: '[]',
        parametrization: '{}',
        version: 0,
        sourceDashboardLink: '',
        owners: [],
      };
    }

    return {
      serviceDashboardId: serviceDashboard.id || '',
      name: serviceDashboard.name || '',
      description: serviceDashboard.description || '',
      widgets: JSON.stringify(serviceDashboard.widgets, null, 2) || '[]',
      parametrization: JSON.stringify(serviceDashboard.parametrization, null, 2) || '{"selectors":"","parameters":[]}',
      version: serviceDashboard.version || 0,
      sourceDashboardLink: serviceDashboard.sourceDashboardLink || '',
      owners: serviceDashboard.owners || [],
    };
  };

  static stateToServiceDashboardUpdate = (state) => ({
    serviceDashboardId: state.serviceDashboardId || '',
    name: state.name || '',
    description: state.description || '',
    widgets: tryParse(state.widgets, []),
    parametrization: tryParse(state.parametrization, {}),
    version: state.version,
    sourceDashboardLink: state.sourceDashboardLink,
    owners: state.owners,
  });

  constructor(props) {
    super(props);
    this.state = ServiceDashboardForm.serviceDashboardToState(props.serviceDashboard, props.isNew);
  }

  componentDidUpdate(prevProps) {
    if (this.props.serviceDashboard !== prevProps.serviceDashboard) {
      const newState = ServiceDashboardForm.serviceDashboardToState(
        this.props.serviceDashboard,
        this.props.isNew,
      );
      this.setState(newState);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const value = inputTargetValue(target);
    this.setState({ [target.name]: value });
  };

  onWidgetsChange = (widgets) => {
    this.setState({ widgets });
  };

  onParametrizationChange = (parametrization) => {
    this.setState({ parametrization });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(ServiceDashboardForm.stateToServiceDashboardUpdate(this.state));
  };

  onDeleteOwner = (index) => {
    this.setState((state) => {
      const newValues = state.owners.filter((r, i) => i !== index);
      return { owners: newValues };
    });
  };

  onUpdateOwner = (index, owner) => {
    this.setState((state) => {
      const newValues = [...state.owners];
      if (index >= 0) {
        newValues[index] = owner;
      } else {
        newValues.push(owner);
      }
      return { owners: newValues };
    });
  };

  render() {
    const {
      name, description, widgets, parametrization, sourceDashboardLink, owners,
    } = this.state;

    return (
      <form className="form-horizontal">
        <div className="col-lg-6 col-md-7">
          <FormInput
            type="text" name="name" label="Name" value={name} onChange={this.onInputChange}
            help="Service dashboard name"
          />
          <FormArea
            label="Description"
            name="description"
            value={description}
            onChange={this.onInputChange}
          />
          <FormInput
            type="text" name="sourceDashboardLink" label="Source link" value={sourceDashboardLink} onChange={this.onInputChange}
            help="Service dashboard source link"
          />
          <FormElement label="Owners">
            <TableEditor
              columns={['Owner']}
              values={owners}
              readonlyRow={ServiceDashboardOwnerTable.ReadonlyRow}
              editableRow={ServiceDashboardOwnerTable.EditableRow}
              onDelete={this.onDeleteOwner}
              onUpdate={this.onUpdateOwner}
            />
            <span className="help-block">
              Dashboard owners
            </span>
          </FormElement>
          <FormElement label="Parametrization JSON">
            <ProtoTextEditor
              name="parametrization"
              value={parametrization}
              onChange={this.onParametrizationChange}
            />
          </FormElement>
          <FormElement label="Widgets JSON">
            <ProtoTextEditor
              name="widgets"
              value={widgets}
              rows={20}
              onChange={this.onWidgetsChange}
            />
          </FormElement>
          <FormButtons onSubmit={this.onSubmit} />
        </div>
      </form>
    );
  }
}

ServiceDashboardForm.propTypes = {
  serviceDashboard: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
};

export default ServiceDashboardForm;
