import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import JSONPretty from 'react-json-pretty';

import PageTitle from '../../components/PageTitle';
import WarningAlert from '../projects/alerts/WarningAlert';
import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import Panel from '../../components/Panel';
import ShowActions from '../../components/ShowActions';
import LabeledValue from '../../components/LabeledValue';
import {
  clearServiceDashboard,
  deleteServiceDashboard,
  loadServiceDashboard,
  validateServiceDashboard,
  validateServiceDashboardSync,
  syncServiceDashboard,
} from '../../store/reducers/serviceProviders/serviceDashboard';
import EntityInfo from '../../components/EntityInfo';
import InputDashboardIdModal from './InputDashboardIdModal';
import DashboardValidationResultModal from './DashboardValidationResultModal';
import { isCloud, isPreEnv, isProdEnv } from '../../utils/env';
import Auth from '../../auth/Auth';
import { ADMIN } from '../../auth/Roles';
import ServiceDashboardWithName from './ServiceDashboardWithName';
import InternalOrExternalLink from '../../components/InternalOrExternalLink';
import TableViewer from '../../components/TableViewer';
import { ReadonlyRow as ServiceDashboardOwnerRow } from './ServiceDashboardOwnerTable';

class ServiceDashboardPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { isOpen: false, isOpenValidation: false, continueSync: undefined };
  }

  componentDidMount() {
    const { serviceDashboardId } = this.props.match.params;
    this.props.loadServiceDashboard(serviceDashboardId);
  }

  componentWillUnmount() {
    this.props.clearServiceDashboard();
  }

  handleDelete = (event) => {
    event.preventDefault();
    const { match, history } = this.props;
    const { serviceProviderId, serviceDashboardId } = match.params;
    this.props.deleteServiceDashboard(serviceDashboardId)
      .then((action) => {
        if (action) {
          history.push(`/admin/serviceProviders/${serviceProviderId}/serviceDashboards`);
        }
      });
  };

  handleCopyFrom = (dashboardId) => {
    const { serviceDashboardId } = this.props.match.params;
    this.props.validateServiceDashboardSync(serviceDashboardId, dashboardId)
      .then(() => {
        const { validationResultsData } = this.props;
        if (validationResultsData.validationIssues.length === 0) {
          this.setState({ isOpen: false }, () => {
            const link = this.makeLink(dashboardId);
            this.props.syncServiceDashboard(serviceDashboardId, dashboardId, link);
          });
        } else {
          this.setState({ isOpenValidation: true, continueSync: dashboardId });
        }
      });
  };

  handleShowDashboardIdModal = () => {
    this.setState({ isOpen: true });
  };

  handleShowValidationModal = () => {
    const { serviceDashboardData } = this.props;
    const data = serviceDashboardData.data;
    const { serviceProviderId } = this.props.match.params;
    this.setState({ isOpenValidation: true }, () => {
      this.props.validateServiceDashboard({ ...data, service: serviceProviderId });
    });
  };

  handleHideDashboardIdModal = () => {
    this.setState({ isOpen: false });
  };

  handleHideValidationModal = () => {
    this.setState({ isOpenValidation: false, continueSync: undefined });
  };

  handleConfirmValidationModal = () => {
    const { continueSync } = this.state;
    if (continueSync !== undefined) {
      const { serviceDashboardId } = this.props.match.params;
      const link = this.makeLink(continueSync);
      this.props.syncServiceDashboard(serviceDashboardId, continueSync, link);
      this.setState({ isOpenValidation: false, continueSync: undefined, isOpen: false });
    } else {
      this.setState({ isOpenValidation: false, continueSync: undefined });
    }
  };

  makeLink = (dashboardId) => {
    if (isPreEnv()) {
      return `https://monitoring-preprod.cloud.yandex.ru/folders/{folderId}/dashboards/${dashboardId}`;
    }
    return `https://monitoring.cloud.yandex.ru/folders/{folderId}/dashboards/${dashboardId}`;
  }

  render() {
    const { serviceProviderId, serviceDashboardId } = this.props.match.params;

    const { serviceDashboardData } = this.props;

    const isCloudProd = isCloud() && isProdEnv();
    const isAdmin = Auth.hasRole(ADMIN);
    const canModify = isAdmin || !isCloudProd;

    if (serviceDashboardData.loading) {
      return <div>Loading...</div>;
    }

    if (serviceDashboardData.error) {
      return <WarningAlert title="Error!" message={serviceDashboardData.error} />;
    }

    if (serviceDashboardData.data) {
      const serviceDashboard = serviceDashboardData.data;

      let copyFromField = null;

      if (canModify) {
        const copyFromButton = (
          <>
            <InputDashboardIdModal
              isOpen={this.state.isOpen}
              onConfirm={this.handleCopyFrom}
              onCancel={this.handleHideDashboardIdModal}
            />
            <button type="button" className="btn btn-default" onClick={this.handleShowDashboardIdModal}>
              Copy from existing dashboard
            </button>
          </>
        );
        copyFromField = (
          <LabeledValue label="" value={copyFromButton} />
        );
      }

      const validateButton = (
        <>
          <DashboardValidationResultModal
            isOpenValidation={this.state.isOpenValidation}
            onConfirm={this.handleConfirmValidationModal}
            onCancel={this.handleHideValidationModal}
          />
          <button type="button" className="btn btn-default" onClick={this.handleShowValidationModal}>
            Validate dashboard
          </button>
        </>
      );
      const validateField = (
        <LabeledValue label="" value={validateButton} />
      );
      const name = serviceDashboard.sourceDashboardLink && serviceDashboard.sourceDashboardLink.substring(serviceDashboard.sourceDashboardLink.lastIndexOf('/') + 1);
      const sourceDashboardLink = (
        <ServiceDashboardWithName name="Source dashboard link">
          <InternalOrExternalLink href={serviceDashboard.sourceDashboardLink}>
            {name}
          </InternalOrExternalLink>
        </ServiceDashboardWithName>
      );

      const ownersContent = serviceDashboard.owners && serviceDashboard.owners.length > 0 ? (
        <TableViewer
          columns={['Owner']}
          values={serviceDashboard.owners}
          row={ServiceDashboardOwnerRow}
          limit={3}
        />
      ) : null;

      return (
        <div>
          <Breadcrumb match={this.props.match} />
          <PageTitle title={`Service dashboard ${serviceDashboard.name}`} />
          <div className="row">
            <div className="col-lg-6">
              <Panel title="Generic options">
                <LabeledValue label="ID" value={serviceDashboard.id} />
                {!!serviceDashboard.description && <LabeledValue label="Description" value={serviceDashboard.description} />}
                {copyFromField}
                {validateField}
                {!!ownersContent && <LabeledValue label="Owners" value={ownersContent} />}
                {!!serviceDashboard.sourceDashboardLink && <LabeledValue label="Source dashboard" value={sourceDashboardLink} />}
                <LabeledValue label="Parametrization" value={<JSONPretty json={serviceDashboard.parametrization} />} />
                <LabeledValue label="Widgets" value={<JSONPretty json={serviceDashboard.widgets} />} />
                <EntityInfo entity={serviceDashboard} />
              </Panel>
              <ShowActions
                editPath={`/admin/serviceProviders/${serviceProviderId}/serviceDashboards/${serviceDashboardId}/edit`}
                duplicatePath=""
                canEdit={canModify}
                canDelete={canModify}
                onDelete={this.handleDelete}
                onDuplicate={false}
                canDuplicate={false}
                canChangeState={false}
              />
            </div>
          </div>
        </div>
      );
    }

    return null;
  }
}

ServiceDashboardPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  serviceDashboardData: PropTypes.object.isRequired,
  validationResultsData: PropTypes.object,
  loadServiceDashboard: PropTypes.func.isRequired,
  syncServiceDashboard: PropTypes.func.isRequired,
  deleteServiceDashboard: PropTypes.func.isRequired,
  clearServiceDashboard: PropTypes.func.isRequired,
  validateServiceDashboard: PropTypes.func.isRequired,
  validateServiceDashboardSync: PropTypes.func.isRequired,
};

ServiceDashboardPage.defaultProps = {
  validationResultsData: {
    validationIssues: [],
  },
};

const mapStateToProps = (state) => ({
  serviceDashboardData: state.serviceDashboard,
  validationResultsData: state.serviceDashboardValidationResult,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadServiceDashboard,
  syncServiceDashboard,
  validateServiceDashboard,
  validateServiceDashboardSync,
  deleteServiceDashboard,
  clearServiceDashboard,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(ServiceDashboardPage);
