import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';

import PageTitle from '../../components/PageTitle';
import WarningAlert from '../projects/alerts/WarningAlert';

import { loadServiceDashboards, clearServiceDashboards } from '../../store/reducers/serviceProviders/serviceDashboards';
import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';

class ServiceDashboardsPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { text: '' };
  }

  componentDidMount() {
    this.textFilter.focus();
    const { serviceProviderId } = this.props.match.params;
    this.props.loadServiceDashboards({ service: serviceProviderId });
  }

  componentWillUnmount() {
    this.props.clearServiceDashboards();
  }

  handleFilterChange = (event) => {
    const { serviceProviderId } = this.props.match.params;
    const text = event.target.value;
    this.setState({ text }, () => {
      this.props.loadServiceDashboards({ service: serviceProviderId, filter: this.state.text });
    });
  };

  render() {
    const { serviceDashboardsPage } = this.props;
    const { serviceProviderId } = this.props.match.params;

    let content = null;

    if (serviceDashboardsPage.loading) {
      content = <div>Loading...</div>;
    } else if (serviceDashboardsPage.error) {
      content = (
        <WarningAlert title="Error!" message="Failed to load service dashboards" />
      );
    } else if (serviceDashboardsPage.data) {
      const serviceDashboards = serviceDashboardsPage.data.serviceDashboards || [];

      content = (
        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>#</th>
              <th>Name</th>
              <th>Description</th>
            </tr>
          </thead>
          <tbody>
            {serviceDashboards.map((serviceDashboard, i) => (
              <tr key={serviceDashboard.id}>
                <td>{i + 1}</td>
                <td><Link to={`/admin/serviceProviders/${serviceProviderId}/serviceDashboards/${serviceDashboard.id}`}>{serviceDashboard.name}</Link></td>
                <td>{serviceDashboard.description}</td>
              </tr>
            ))}
          </tbody>
        </table>
      );
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title="Service dashboards" />
        <div className="row">
          <div className="btn-toolbar table-toolbar">
            <div className="btn-group">
              <Link to={`/admin/serviceProviders/${serviceProviderId}/serviceDashboards/new`} className="btn btn-outline btn-success">
                <i className="glyphicon glyphicon-plus" />
                {' '}
                Create New
              </Link>
            </div>
            <div className="btn-group">
              <input
                type="text"
                className="form-control"
                value={this.state.text}
                onChange={this.handleFilterChange}
                ref={(input) => { this.textFilter = input; }}
                placeholder="Type to filter..."
              />
            </div>
          </div>
        </div>

        <div className="row">
          {content}
        </div>
      </div>
    );
  }
}

ServiceDashboardsPage.propTypes = {
  match: PropTypes.object.isRequired,
  serviceDashboardsPage: PropTypes.object.isRequired,
  loadServiceDashboards: PropTypes.func.isRequired,
  clearServiceDashboards: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({ serviceDashboardsPage: state.serviceDashboardsPage });

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadServiceDashboards,
  clearServiceDashboards,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(ServiceDashboardsPage);
