import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import * as AggrRules from './AggrRulesTable';
import * as References from './ReferencesTable';
import * as ServiceAccounts from './ServiceAccountsTable';
import TableEditor from '../../components/TableEditor';
import FormButtons from '../../components/forms/FormButtons';
import FormInput from '../../components/forms/FormInput';
import FormCheckbox from '../../components/forms/FormCheckbox';
import FormElement from '../../components/forms/FormElement';
import { inputTargetValue } from '../../utils/inputs';
import FormArea from '../../components/forms/FormArea';

class ServiceProviderForm extends PureComponent {
  static serviceProviderToState = (serviceProvider, isNew) => {
    if (isNew) {
      return {
        serviceProviderId: '',
        references: [],
        description: '',
        aggrRules: [],
        memOnly: false,
        metricTtlDays: 0,
        gridSeconds: 0,
        intervalSeconds: 0,
        abcService: '',
        cloudId: '',
        tvmDestId: '',
        iamServiceAccountIds: [],
        tvmServiceIds: [],
        version: 0,
      };
    }
    const shardSettings = serviceProvider.shardSettings || {};
    return {
      serviceProviderId: serviceProvider.id || '',
      description: serviceProvider.description || '',
      references: serviceProvider.references || [],
      aggrRules: shardSettings.aggrRules || [],
      memOnly: !!shardSettings.memOnly,
      metricTtlDays: shardSettings.metricTtlDays || 0,
      gridSeconds: shardSettings.gridSeconds || 0,
      intervalSeconds: shardSettings.intervalSeconds || 0,
      abcService: serviceProvider.abcService || '',
      cloudId: serviceProvider.cloudId || '',
      tvmDestId: serviceProvider.tvmDestId || '',
      iamServiceAccountIds: serviceProvider.iamServiceAccountIds || [],
      tvmServiceIds: serviceProvider.tvmServiceIds || [],
      version: serviceProvider.version || 0,
    };
  };

  static stateToServiceProviderUpdate = (state) => ({
    serviceProviderId: state.serviceProviderId || '',
    description: state.description || '',
    references: state.references || [],
    shardSettings: {
      aggrRules: state.aggrRules || [],
      memOnly: state.memOnly,
      metricTtlDays: state.metricTtlDays,
      gridSeconds: state.gridSeconds,
      intervalSeconds: state.intervalSeconds,
    },
    abcService: state.abcService,
    cloudId: state.cloudId,
    tvmDestId: state.tvmDestId,
    iamServiceAccountIds: state.iamServiceAccountIds || [],
    tvmServiceIds: state.tvmServiceIds || [],
    version: state.version,
  });

  constructor(props) {
    super(props);
    this.state = ServiceProviderForm.serviceProviderToState(props.serviceProvider, props.isNew);
  }

  componentDidUpdate(prevProps) {
    if (this.props.serviceProvider !== prevProps.serviceProvider) {
      const newState = ServiceProviderForm.serviceProviderToState(
        this.props.serviceProvider,
        this.props.isNew,
      );
      this.setState(newState);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const value = inputTargetValue(target);
    this.setState({ [target.name]: value });
  };

  onDeleteAggrRule = (index) => {
    this.setState((state) => {
      const newAggrRules = state.aggrRules.filter((r, i) => i !== index);
      return { aggrRules: newAggrRules };
    });
  };

  onUpdateAggrRule = (index, rule) => {
    this.setState((state) => {
      const newAggrRules = [...state.aggrRules];
      if (index >= 0) {
        newAggrRules[index] = rule;
      } else {
        newAggrRules.push(rule);
      }
      return { aggrRules: newAggrRules };
    });
  };

  onDeleteReference = (index) => {
    this.setState((state) => {
      const newReferences = state.references.filter((r, i) => i !== index);
      return { references: newReferences };
    });
  };

  onUpdateReference = (index, reference) => {
    this.setState((state) => {
      const newReferences = [...state.references];
      if (index >= 0) {
        newReferences[index] = reference;
      } else {
        newReferences.push(reference);
      }
      return { references: newReferences };
    });
  };

  onDeleteServiceAccountId = (index) => {
    this.setState((state) => {
      const newValues = state.iamServiceAccountIds.filter((r, i) => i !== index);
      return { iamServiceAccountIds: newValues };
    });
  };

  onDeleteTvmServiceId = (index) => {
    this.setState((state) => {
      const newValues = state.tvmServiceIds.filter((r, i) => i !== index);
      return { tvmServiceIds: newValues };
    });
  };

  onUpdateServiceAccountId = (index, serviceAccountId) => {
    this.setState((state) => {
      const newValues = [...state.iamServiceAccountIds];
      if (index >= 0) {
        newValues[index] = serviceAccountId;
      } else {
        newValues.push(serviceAccountId);
      }
      return { iamServiceAccountIds: newValues };
    });
  };

  onUpdateTvmServiceId = (index, serviceId) => {
    this.setState((state) => {
      const newValues = [...state.tvmServiceIds];
      if (index >= 0) {
        newValues[index] = serviceId;
      } else {
        newValues.push(serviceId);
      }
      return { tvmServiceIds: newValues };
    });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(ServiceProviderForm.stateToServiceProviderUpdate(this.state));
  };

  render() {
    const {
      serviceProviderId, description, references, aggrRules, memOnly, metricTtlDays, gridSeconds,
      intervalSeconds, abcService, cloudId, tvmDestId, iamServiceAccountIds, tvmServiceIds,
    } = this.state;

    return (
      <form className="form-horizontal">
        <div className="col-lg-6 col-md-7">
          <FormInput
            type="text" name="serviceProviderId" label="Id" value={serviceProviderId} onChange={this.onInputChange}
            help="Unique service provider ID (equal to name of created services)."
            disabled={!this.props.isNew}
          />
          <FormArea
            label="Description"
            name="description"
            value={description}
            onChange={this.onInputChange}
          />
          <FormElement label="References">
            <TableEditor
              columns={['#', 'Label', 'Services', 'Types', 'Cross folder']}
              values={references}
              readonlyRow={References.ReadonlyRow}
              editableRow={References.EditableRow}
              onDelete={this.onDeleteReference}
              onUpdate={this.onUpdateReference}
            />
          </FormElement>
          <FormElement label="Aggregation rules">
            <TableEditor
              columns={['#', 'Conditions', 'Targets', 'Function']}
              values={aggrRules}
              readonlyRow={AggrRules.ReadonlyRow}
              editableRow={AggrRules.EditableRow}
              onDelete={this.onDeleteAggrRule}
              onUpdate={this.onUpdateAggrRule}
            />
            <span className="help-block">
              Aggregates configured in next form:
              {' '}
              <b>MATCH_CONDITION</b>
              &nbsp;-&gt;&nbsp;
              <b>LABELS_EXPRESSION</b>
              . Match condition and labels expression are
              list of labels (written in &quot;key=value&quot; format) separated
              by comma (,). Read more info on the&nbsp;
              <a href="https://wiki.yandex-team.ru/solomon/userguide/aggregates/" target="_blank" rel="noopener noreferrer">wiki page</a>
              .
            </span>
          </FormElement>
          <FormInput
            type="number"
            name="metricTtlDays"
            label="Metrics TTL"
            value={metricTtlDays}
            onChange={this.onInputChange}
            suffix="days"
            // eslint-disable-next-line max-len
            help="If this field is filled, Solomon will drop all metrics where last point was metrics TTL days ago"
          />
          <FormInput
            type="number" name="gridSeconds" label="Grid" value={gridSeconds} onChange={this.onInputChange}
            help="Defines data granularity, must be multiple of 5m. Can be different from fetch interval,
                  in cases when fetch\push multiple points by one time Default value equal to Interval.
                  You can leave this field empty to use default value"
          />
          <FormInput
            type="number" name="intervalSeconds" label="Interval" value={intervalSeconds} onChange={this.onInputChange}
            help="Define interval to fetch metrics from agents"
          />
          <FormCheckbox
            name="memOnly" label="Store only aggregates"
            value={memOnly}
            onChange={this.onInputChange}
            help="If true Solomon will not store all fetched raw data. Only configured
                  aggregates (see tab 'Aggregation rules') will be stored"
          />
          <FormInput
            type="text"
            name="cloudId"
            label="Cloud ID"
            value={cloudId}
            onChange={this.onInputChange}
            help="Cloud ID of service provider's internal cloud. Needed for finding responsibles"
          />
          <FormElement label="IAM service account IDs">
            <TableEditor
              columns={['#', 'Account ID']}
              values={iamServiceAccountIds}
              readonlyRow={ServiceAccounts.ReadonlyRow}
              editableRow={ServiceAccounts.EditableRow}
              onDelete={this.onDeleteServiceAccountId}
              onUpdate={this.onUpdateServiceAccountId}
            />
            <span className="help-block">
              IAM service account ID of your application
              that Solomon will contact to get metrics from
            </span>
          </FormElement>
          <FormInput
            type="text"
            name="abcService"
            label="ABC service"
            value={abcService}
            onChange={this.onInputChange}
            help="ABC service of service provider. Needed for finding responsibles. Not needed in Cloud environment"
          />
          <FormInput
            type="text"
            name="tvmDestId"
            label="TVM destination ID"
            value={tvmDestId}
            onChange={this.onInputChange}
            help="TVM clent ID of your application that Solomon will contact to push metrics from"
          />
          <FormElement label="TVM service IDs">
            <TableEditor
              columns={['#', 'TVM Service ID']}
              values={tvmServiceIds}
              readonlyRow={ServiceAccounts.ReadonlyRow}
              editableRow={ServiceAccounts.EditableRow}
              onDelete={this.onDeleteTvmServiceId}
              onUpdate={this.onUpdateTvmServiceId}
            />
            <span className="help-block">
              TVM service ID of your application
              that Solomon will contact to get metrics from
            </span>
          </FormElement>
          <FormButtons onSubmit={this.onSubmit} />
        </div>
      </form>
    );
  }
}

ServiceProviderForm.propTypes = {
  serviceProvider: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
};

export default ServiceProviderForm;
