import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import { Link } from 'react-router-dom';

import PageTitle from '../../components/PageTitle';
import WarningAlert from '../projects/alerts/WarningAlert';
import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import Panel from '../../components/Panel';
import ShowActions from '../../components/ShowActions';
import LabeledValue from '../../components/LabeledValue';
import { ReadonlyRow as AggrRuleRow } from './AggrRulesTable';
import { ReadonlyRow as ReferenceRow } from './ReferencesTable';
import { ReadonlyRow as ServiceAccountRow } from './ServiceAccountsTable';
import TableViewer from '../../components/TableViewer';
import {
  clearServiceProvider,
  deleteServiceProvider,
  loadServiceProvider,
} from '../../store/reducers/serviceProviders/serviceProvider';
import EntityInfo from '../../components/EntityInfo';
import { ADMIN } from '../../auth/Roles';
import Auth from '../../auth/Auth';

class ServiceProviderPage extends PureComponent {
  componentDidMount() {
    const { serviceProviderId } = this.props.match.params;
    this.props.loadServiceProvider(serviceProviderId);
  }

  componentWillUnmount() {
    this.props.clearServiceProvider();
  }

  handleDelete = (event) => {
    event.preventDefault();
    const { match, history } = this.props;
    const { serviceProviderId } = match.params;
    this.props.deleteServiceProvider(serviceProviderId)
      .then((action) => {
        if (action) {
          history.push('/admin/serviceProviders');
        }
      });
  };

  render() {
    const { serviceProviderId } = this.props.match.params;

    const { serviceProviderData } = this.props;

    const isAdmin = Auth.hasRole(ADMIN);

    if (serviceProviderData.loading) {
      return <div>Loading...</div>;
    }

    if (serviceProviderData.error) {
      return <WarningAlert title="Error!" message={serviceProviderData.error} />;
    }

    if (serviceProviderData.data) {
      const serviceProvider = serviceProviderData.data;

      const { references, iamServiceAccountIds, tvmServiceIds } = serviceProvider;

      const {
        aggrRules, memOnly, metricTtlDays, gridSeconds, intervalSeconds,
      } = serviceProvider.shardSettings;

      const aggrRulesContent = aggrRules && aggrRules.length > 0 ? (
        <TableViewer
          columns={['#', 'Conditions', 'Targets', 'Function', '']}
          values={aggrRules}
          row={AggrRuleRow}
          limit={3}
        />
      ) : null;

      const gridValue = gridSeconds ? `${gridSeconds} seconds` : '0';
      const intervalValue = intervalSeconds ? `${intervalSeconds} seconds` : '0';

      const referencesContent = references && references.length > 0 ? (
        <TableViewer
          columns={['#', 'Label', 'Services', 'Types', 'Cross folder']}
          values={references}
          row={ReferenceRow}
          limit={3}
        />
      ) : null;

      const serviceAccountsContent = iamServiceAccountIds && iamServiceAccountIds.length > 0 ? (
        <TableViewer
          columns={['#', 'Account ID']}
          values={iamServiceAccountIds}
          row={ServiceAccountRow}
          limit={3}
        />
      ) : null;

      const tvmServicesContent = tvmServiceIds && tvmServiceIds.length > 0 ? (
        <TableViewer
          columns={['#', 'TVM Service ID']}
          values={tvmServiceIds}
          row={ServiceAccountRow}
          limit={3}
        />
      ) : null;

      return (
        <div>
          <Breadcrumb match={this.props.match} />
          <PageTitle title={`Service provider ${serviceProviderId}`} />
          <div className="row">
            <div className="col-lg-6">
              <Panel title="Generic options">
                {!!serviceProvider.description && <LabeledValue label="Description" value={serviceProvider.description} />}
                {!!referencesContent && <LabeledValue label="References" value={referencesContent} />}
                {!!aggrRulesContent && <LabeledValue label="Aggregation rules" value={aggrRulesContent} />}
                {!!memOnly && <LabeledValue label="Store only aggregates" value="On" />}
                {metricTtlDays
                  ? <LabeledValue label="Metrics TTL" value={`${metricTtlDays} days`} />
                  : null}
                <LabeledValue label="Grid" value={gridValue} />
                <LabeledValue label="Interval" value={intervalValue} />
                {!!serviceProvider.abcService
                  && (
                  <LabeledValue
                    label="ABC service"
                    value={<a href={`https://abc.yandex-team.ru/services/${serviceProvider.abcService}`}>{serviceProvider.abcService}</a>}
                  />
                  )}
                {!!serviceProvider.cloudId && <LabeledValue label="Cloud ID" value={serviceProvider.cloudId} />}
                {!!serviceProvider.tvmDestId && <LabeledValue label="TVM Destination ID" value={serviceProvider.tvmDestId} />}
                {!!serviceAccountsContent && <LabeledValue label="IAM service account IDs" value={serviceAccountsContent} />}
                {!!tvmServicesContent && <LabeledValue label="TVM service IDs" value={tvmServicesContent} />}
                {!!serviceProvider.cloudId && <LabeledValue label="" value={<Link to={`/admin/serviceProviders/${serviceProviderId}/serviceDashboards`}>Service dashboards</Link>} />}
                <LabeledValue label="" value={<Link to={`/admin/serviceProviders/${serviceProviderId}/agents`}>Agents</Link>} />
                <EntityInfo entity={serviceProvider} />
              </Panel>
              <ShowActions
                editPath={`/admin/serviceProviders/${serviceProviderId}/edit`}
                duplicatePath=""
                onDelete={this.handleDelete}
                onDuplicate={false}
                canChangeState={false}
                canDelete={isAdmin}
                canDuplicate={false}
                canEdit={isAdmin}
              />
            </div>
          </div>
        </div>
      );
    }

    return null;
  }
}

ServiceProviderPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  serviceProviderData: PropTypes.object.isRequired,
  loadServiceProvider: PropTypes.func.isRequired,
  deleteServiceProvider: PropTypes.func.isRequired,
  clearServiceProvider: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  serviceProviderData: state.serviceProvider,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadServiceProvider,
  deleteServiceProvider,
  clearServiceProvider,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(ServiceProviderPage);
