import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';

import PageTitle from '../../components/PageTitle';
import WarningAlert from '../projects/alerts/WarningAlert';

import { loadServiceProviders, clearServiceProviders } from '../../store/reducers/serviceProviders/serviceProviders';
import { ADMIN } from '../../auth/Roles';
import Auth from '../../auth/Auth';
import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';

class ServiceProvidersPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { text: '' };
  }

  componentDidMount() {
    this.textFilter.focus();
    this.props.loadServiceProviders({});
  }

  componentWillUnmount() {
    this.props.clearServiceProviders();
  }

  handleFilterChange = (event) => {
    const text = event.target.value;
    this.setState({ text }, () => {
      this.props.loadServiceProviders({ filter: this.state.text });
    });
  };

  render() {
    const { serviceProvidersPage } = this.props;

    let content = null;

    if (serviceProvidersPage.loading) {
      content = <div>Loading...</div>;
    } else if (serviceProvidersPage.error) {
      content = (
        <WarningAlert title="Error!" message="Failed to load service providers" />
      );
    } else if (serviceProvidersPage.data) {
      const serviceProviders = serviceProvidersPage.data.serviceProviders || [];

      content = (
        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>#</th>
              <th>ID</th>
              {/* <th>ABC service</th> */}
              <th>Cloud ID</th>
              <th>IAM Accounts</th>
              <th>TVM Services</th>
              <th>References</th>
              <th>TTL</th>
              <th>Grid</th>
              <th>Interval</th>
              <th>Rules</th>
              <th>MemOnly</th>
              <th>Description</th>
            </tr>
          </thead>
          <tbody>
            {serviceProviders.map((serviceProvider, i) => (
              <tr key={serviceProvider.id}>
                <td>{i + 1}</td>
                <td><Link to={`/admin/serviceProviders/${serviceProvider.id}`}>{serviceProvider.id}</Link></td>
                {/* <td>{serviceProvider.abcService ? <a href={`https://abc.yandex-team.ru/services/${serviceProvider.abcService}`}>{serviceProvider.abcService}</a> : '-'}</td> */}
                <td>{serviceProvider.cloudId || '-'}</td>
                <td>{serviceProvider.iamServiceAccountIds.length}</td>
                <td>{serviceProvider.tvmServiceIds.length}</td>
                <td>{serviceProvider.references.length}</td>
                <td>
                  {serviceProvider.shardSettings.metricTtlDays}
                  {'d'}
                </td>
                <td>
                  {serviceProvider.shardSettings.gridSeconds}
                  {'s'}
                </td>
                <td>
                  {serviceProvider.shardSettings.intervalSeconds}
                  {'s'}
                </td>
                <td>{serviceProvider.shardSettings.aggrRules.length}</td>
                <td>{serviceProvider.shardSettings.memOnly ? 'On' : 'Off'}</td>
                <td>{serviceProvider.description}</td>
              </tr>
            ))}
          </tbody>
        </table>
      );
    }

    const isAdmin = Auth.hasRole(ADMIN);

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title="Service providers" />
        <div className="row">
          <div className="btn-toolbar table-toolbar">
            {isAdmin && (
              <div className="btn-group">
                <Link to="/admin/serviceProviders/new" className="btn btn-outline btn-success">
                  <i className="glyphicon glyphicon-plus" />
                  {' '}
                  Create New
                </Link>
              </div>
            )}
            <div className="btn-group">
              <input
                type="text"
                className="form-control"
                value={this.state.text}
                onChange={this.handleFilterChange}
                ref={(input) => { this.textFilter = input; }}
                placeholder="Type to filter..."
              />
            </div>
          </div>
        </div>

        <div className="row">
          {content}
        </div>
      </div>
    );
  }
}

ServiceProvidersPage.propTypes = {
  match: PropTypes.object.isRequired,
  serviceProvidersPage: PropTypes.object.isRequired,
  loadServiceProviders: PropTypes.func.isRequired,
  clearServiceProviders: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({ serviceProvidersPage: state.serviceProvidersPage });

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadServiceProviders,
  clearServiceProviders,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(ServiceProvidersPage);
