/* eslint-disable max-classes-per-file */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import LabelsList from '../../components/LabelsList';

const RULE_SHAPE = PropTypes.shape({
  cond: PropTypes.arrayOf(PropTypes.string),
  target: PropTypes.arrayOf(PropTypes.string),
  function: PropTypes.string,
});

export const AGGREGATION_FUNCTION_OPTIONS = [
  { value: 'FUNCTION_UNSPECIFIED', title: 'default' },
  { value: 'SUM', title: 'sum' },
  { value: 'LAST', title: 'last' },
];

const getTitleByValue = (value, options) => {
  const filteredOptions = options.filter((o) => o.value === value || o.value === String(value));
  if (filteredOptions.length !== 1) {
    return '';
  }
  return filteredOptions[0].title;
};

function removeDefaultAgregation(id) {
  if (id === 'FUNCTION_UNSPECIFIED' || id === '') {
    return undefined;
  }
  return id;
}

function wrapHandler(index, handler) {
  if (!handler) {
    return null;
  }
  return (event) => {
    event.preventDefault();
    event.stopPropagation();
    handler(index);
  };
}

export class ReadonlyRow extends PureComponent {
  constructor(props) {
    super(props);
    this.onClick = wrapHandler(props.index, props.onClick);
    this.onDelete = wrapHandler(props.index, props.onDelete);
  }

  render() {
    const { index, value } = this.props;
    return (
      <tr onClick={this.onClick}>
        <td>{index + 1}</td>
        <td>
          <LabelsList values={value.cond || []} type="success" />
        </td>
        <td>
          <LabelsList values={value.target || []} type="info" />
        </td>
        <td>{getTitleByValue(value.function, AGGREGATION_FUNCTION_OPTIONS)}</td>
        {this.onDelete && (
          <td className="actions">
            <button type="button" onClick={this.onDelete}>
              <i className="glyphicon glyphicon-remove" />
            </button>
          </td>
        )}
      </tr>
    );
  }
}

ReadonlyRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: RULE_SHAPE.isRequired,
  onClick: PropTypes.func,
  onDelete: PropTypes.func,
};

ReadonlyRow.defaultProps = {
  onClick: null,
  onDelete: null,
};

function listToStr(list) {
  return list ? list.join(', ') : '';
}

function strToList(str) {
  return str ? str.split(/\s*,\s*/) : [];
}

export class EditableRow extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      cond: listToStr(props.value.cond),
      target: listToStr(props.value.target),
      fn: props.value.function,
    };
  }

  componentDidMount() {
    this.condInput.focus();
  }

  onOkClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.doComplete();
  };

  onKeyDown = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      event.stopPropagation();
      this.doComplete();
    } else if (event.which === 27) {
      event.preventDefault();
      this.props.onCancel();
    }
  };

  onInputChange = (event) => {
    const { name, type } = event.target;
    let { value } = event.target;

    if (type === 'number') {
      value = parseInt(value, 10);
    }
    this.setState({ [name]: value });
  };

  onFnChange = (event) => {
    event.preventDefault();
    this.setState({ [event.target.name]: removeDefaultAgregation(event.target.value) });
  };

  doComplete() {
    const { onOk, index } = this.props;
    const { cond, target, fn } = this.state;
    onOk(index, {
      cond: strToList(cond),
      target: strToList(target),
      function: removeDefaultAgregation(fn),
    });
  }

  render() {
    const { index } = this.props;
    const { cond, target, fn } = this.state;
    return (
      <tr onKeyDown={this.onKeyDown}>
        <td>{index >= 0 ? index + 1 : ''}</td>
        <td>
          <input
            type="text" name="cond" className="form-control"
            value={cond} onChange={this.onInputChange}
            ref={(input) => { this.condInput = input; }}
          />
        </td>
        <td>
          <input
            type="text" name="target" className="form-control"
            value={target} onChange={this.onInputChange}
          />
        </td>
        <td>
          <select
            className="form-control"
            name="fn"
            value={fn}
            onChange={this.onFnChange}
          >
            {AGGREGATION_FUNCTION_OPTIONS.map((option, fnIndex) => (
              // eslint-disable-next-line react/no-array-index-key
              <option key={fnIndex} value={option.value}>
                {option.title}
              </option>
            ))}
          </select>
        </td>
        <td className="actions">
          <button type="button" onClick={this.onOkClick}>
            <i className="glyphicon glyphicon-ok" />
          </button>
        </td>
      </tr>
    );
  }
}

EditableRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: RULE_SHAPE,
  onOk: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

EditableRow.defaultProps = {
  value: { cond: [], target: [], function: '' },
};
