import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import ServiceForm from './ServiceForm';
import { parseSearch } from '../../utils/url';
import { wrapSubProjectEditPage } from '../projects/wrapSubProjectPage';

import { findService, saveService, clearService } from '../../store/reducers/services/service';

class ServiceEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._serviceId = props.match.params.serviceId;
    this._isNewService = !this._serviceId;

    if (this._isNewService) {
      this._serviceId = parseSearch(props.location.search).get('duplicateOf');
    }
  }

  componentDidMount() {
    const { service } = this.props;
    if (this._serviceId) {
      const loadedServiceId = service.id;
      if (loadedServiceId !== this._serviceId) {
        this.props.findService(this._projectId, this._serviceId);
      }
    } else {
      this.props.clearService();
    }
  }

  onSubmit = (value) => {
    const { history } = this.props;
    this.props.saveService(this._projectId, value, this._isNewService)
      .then(() => history.push(`/admin/projects/${this._projectId}/services/${value.id}`));
  };

  render() {
    const service = this.props.service || {};
    const title = this._isNewService ? 'New Service' : `Service ${this._serviceId}`;
    const isLoading = (this._serviceId && isEmpty(service));

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {isLoading && <span>Loading...</span>}
        {!isLoading
          && (
          <ServiceForm
            projectId={this._projectId}
            service={service}
            onlySensorNameShards={this.props.onlySensorNameShards}
            isNew={this._isNewService}
            onSubmit={this.onSubmit}
          />
          )}
      </div>
    );
  }
}

ServiceEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  service: PropTypes.object.isRequired,
  onlySensorNameShards: PropTypes.bool.isRequired,
  findService: PropTypes.func.isRequired,
  saveService: PropTypes.func.isRequired,
  clearService: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  service: state.service,
  onlySensorNameShards: state.projectData.project.onlySensorNameShards,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findService,
  saveService,
  clearService,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ServiceEditPage);

export default wrapSubProjectEditPage(connectedPage);
