import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import * as AggrRules from './AggrRulesTable';
import TableEditor from '../../components/TableEditor';
import FormButtons from '../../components/forms/FormButtons';
import FormInput from '../../components/forms/FormInput';
import FormCheckbox from '../../components/forms/FormCheckbox';
import FormElement from '../../components/forms/FormElement';
import RadioButtons from '../../components/RadioButtons';
import { TabPane, Tabs } from '../../components/Tabs';
import ApiCallExample from '../../components/ApiCallExample';
import { inputTargetValue } from '../../utils/inputs';
import FormMetricNameLabel from '../../components/forms/FormMetricNameLabel';
import { PULL_PROTOCOL_CHOICES } from './constants';

function deleteRuleReducer(state, name, index) {
  const { sensorConf } = state;

  const newRules = sensorConf[name].filter((r, i) => i !== index);
  return {
    sensorConf: {
      ...sensorConf,
      [name]: newRules,
    },
  };
}

function updateRuleReducer(state, name, index, rule) {
  const { sensorConf } = state;

  let newRules = [];
  if (index >= 0) {
    newRules = [...sensorConf[name]];
    newRules[index] = rule;
  } else if (sensorConf && (name in sensorConf)) {
    newRules = [...sensorConf[name], rule];
  } else {
    newRules = [rule];
  }

  return {
    sensorConf: {
      ...sensorConf,
      [name]: newRules,
    },
  };
}

const MM_PULL = 'PULL';
const MM_PUSH = 'PUSH';
const MONITORING_MODELS = [MM_PULL, MM_PUSH];

class ServiceForm extends PureComponent {
  static serviceToState = (service, isNew) => {
    if (service.id) {
      const state = { ...service };
      state.id = isNew ? '' : state.id;

      state.model = ('port' in state && state.port !== 0) ? MM_PULL : MM_PUSH;
      return state;
    }
    return { id: '', model: MM_PULL, sensorsTtlDays: '30' };
  };

  static cleanService = (service) => {
    const newService = { ...service };
    if (newService.model !== MM_PULL) {
      delete newService.protocol;
      delete newService.port;
      delete newService.path;
      delete newService.addTsArgs;
    }
    delete newService.model;
    return newService;
  };

  constructor(props) {
    super(props);

    this.state = ServiceForm.serviceToState(props.service, props.isNew);
    this._nameInputTouched = !!props.service.name;
    this._isNewService = props.isNew;
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    // We cannot use setState() because it performs a shallow merge of
    // previous state with the new one. But here we need entirely update state.
    if (this.props.projectId !== prevProps.projectId || this.props.service !== prevProps.service) {
      this.state = ServiceForm.serviceToState(this.props.service);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const value = inputTargetValue(target);

    const change = {};
    switch (target.name) {
      case 'id':
        change.id = `${this.props.projectId}_${value}`;
        if (!this._nameInputTouched) {
          change.name = value;
        }
        break;
      case 'name':
        this._nameInputTouched = true;
        change.name = value;
        break;
      case 'rawDataMemOnly':
        change.sensorConf = { ...this.state.sensorConf };
        change.sensorConf.rawDataMemOnly = value;
        break;
      default:
        change[target.name] = value;
    }

    this.setState(change);
  };

  onModelSelect = (model) => {
    this.setState({ model });
  };

  onProtocolSelect = (protocol) => {
    this.setState({ protocol });
  };

  onDeleteAggrRule = (index) => {
    this.setState(deleteRuleReducer(this.state, 'aggrRules', index));
  };

  onUpdateAggrRule = (index, rule) => {
    this.setState(updateRuleReducer(this.state, 'aggrRules', index, rule));
  };

  onSensorNameLabelChange = (sensorNameLabel) => {
    this.setState({ sensorNameLabel });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(ServiceForm.cleanService(this.state));
  };

  onJsonStateChange = (newState) => {
    this.setState(ServiceForm.serviceToState(newState, false));
    this.forceUpdate();
  };

  render() {
    const { projectId } = this.props;

    const {
      name, protocol, path, port, model, interval, gridSec,
      addTsArgs, sensorConf, sensorsTtlDays, sensorNameLabel, tvmDestId,
    } = this.state;

    let { id } = this.state;
    if (id.startsWith(`${projectId}_`)) {
      id = id.substring(projectId.length + 1);
    }

    let aggrRules = [];
    let rawDataMemOnly = false;
    if (sensorConf) {
      aggrRules = sensorConf.aggrRules || [];
      rawDataMemOnly = sensorConf.rawDataMemOnly || false;
    }

    const intervalHelp = (model === MM_PULL)
      ? 'Defines delay in seconds which Solomon will wait after fetching metrics '
        + 'data from your application and before performing new fetch request'
      : 'Defines estimated interval of metric pushes into Solomon';

    let nameWarning = null;
    if (this._nameInputTouched && !this._isNewService) {
      nameWarning = (
        <div>
          <p>
            <strong>
              <span className="glyphicon glyphicon-alert" />
              {' '}
              Important!
            </strong>
          </p>
          <p>
            Modifying the service name will change all the URLs and metric queries
            (graphs, dashboards in Solomon, Grafana, Statface, etc.)
            related to the service.
          </p>
          <p>
            If you really want to change the name, please write to us about it:
            {' '}
            <a href="mailto://solomon@yandex-team.ru">solomon@yandex-team.ru</a>
          </p>
        </div>
      );
    }

    return (
      <form className="form-horizontal">
        <div className="col-lg-6 col-md-7">
          <Tabs>
            <TabPane label="Generic">
              <FormInput
                type="text" name="id" label="Id" value={id} onChange={this.onInputChange}
                prefix={`${projectId}_`}
                help="Unique service identificator. Project id will be added as prefix"
                disabled={!this._isNewService}
              />
              <FormInput
                type="text" name="name" label="Label name" value={name} onChange={this.onInputChange}
                help="This value will be displayed in selectors in Solomon UI.
                      If you change this value for PUSH service make sure to change
                      value of 'service' label in your pushing data"
                warning={nameWarning}
              />
              <FormMetricNameLabel
                type="service"
                label="Metric name label"
                name="sensorNameLabel"
                value={sensorNameLabel}
                onChange={this.onSensorNameLabelChange}
                onlyMetricNameShards={this.props.onlySensorNameShards}
              />
              <FormElement
                label="Monitoring model"
                help="Which method will be used to send metrics data to Solomon. For PULL
                      model Solomon will fetch metrics data from your application.
                      For PUSH model your application must send metrics data to Solomon"
              >
                <RadioButtons
                  choices={MONITORING_MODELS} selected={model} onSelect={this.onModelSelect}
                />
              </FormElement>
              {model === MM_PULL
                && (
                <FormElement
                  label="Protocol"
                  help="Protocol which will be used by Solomon to fetch metrics from your application"
                >
                  <RadioButtons
                    choices={PULL_PROTOCOL_CHOICES}
                    selected={protocol}
                    onSelect={this.onProtocolSelect}
                  />
                </FormElement>
                )}
              {model === MM_PULL
                && (
                <FormInput
                  type="number" name="port" label="Port" value={port} onChange={this.onInputChange}
                  help="Port which Solomon shoud use to fetch metrics from your application"
                />
                )}
              {model === MM_PULL
                && (
                <FormInput
                  type="text" name="path" label="URL Path" value={path} onChange={this.onInputChange}
                  help="URL on which your application exposes metrics via HTTP"
                />
                )}
              <FormInput
                type="number" name="interval" label="Interval" value={interval} onChange={this.onInputChange}
                help={`${intervalHelp}. Default value is 15 seconds. You can leave this field empty to use default value`}
              />
              <FormInput
                type="number" name="gridSec" label="Grid" value={gridSec} onChange={this.onInputChange}
                help="Defines data granularity, must be multiple of 5m. Can be different from fetch interval,
                      in cases when fetch\push multiple points by one time Default value equal to Interval.
                      You can leave this field empty to use default value"
              />
              {model === MM_PULL
                && (
                <FormCheckbox
                  name="addTsArgs" label="Add timestamp args" value={addTsArgs} onChange={this.onInputChange}
                  help="If true Solomon will add 'now' (e.g. '2017-06-11T00:45:45.542Z') and
                        'period' (e.g. '1d2h3m4s') query parameter into fetch request"
                />
                )}
              <FormInput
                type="number"
                name="sensorsTtlDays"
                label="Metrics TTL"
                value={sensorsTtlDays}
                onChange={this.onInputChange}
                suffix="days"
                // eslint-disable-next-line max-len
                help="If this field is filled, Solomon will drop all metrics where last point was metrics TTL days ago"
              />
              <FormCheckbox
                name="rawDataMemOnly" label="Store only aggregates" value={rawDataMemOnly}
                onChange={this.onInputChange}
                help="If true Solomon will not store all fetched raw data. Only configured
                      aggregates (see tab 'Aggregation rules') will be stored"
              />
              <FormInput
                type="text"
                name="tvmDestId"
                label="TVM destination id"
                value={tvmDestId}
                onChange={this.onInputChange}
                help="TVM clent id of your application that Solomon will contact to download metrics from"
              />
            </TabPane>
            <TabPane label="Aggregation rules">
              <span className="help-block">
                Aggregates configured in next form:
                {' '}
                <b>MATCH_CONDITION</b>
                &nbsp;-&gt;&nbsp;
                <b>LABELS_EXPRESSION</b>
                . Match condition and labels expression are
                list of labels (written in &quot;key=value&quot; format) separated
                by comma (,). Read more info on the&nbsp;
                <a href="https://wiki.yandex-team.ru/solomon/userguide/aggregates/" target="_blank" rel="noopener noreferrer">wiki page</a>
                .
              </span>
              <TableEditor
                columns={['#', 'Conditions', 'Targets', 'Function']}
                values={aggrRules}
                readonlyRow={AggrRules.ReadonlyRow}
                editableRow={AggrRules.EditableRow}
                onDelete={this.onDeleteAggrRule}
                onUpdate={this.onUpdateAggrRule}
              />
            </TabPane>
          </Tabs>
          <FormButtons onSubmit={this.onSubmit} />
        </div>

        <div className="col-lg-6 col-md-5">
          <ApiCallExample
            path={`/projects/${projectId}/services`}
            value={ServiceForm.cleanService(this.state)}
            isNew={this._isNewService}
            onChange={this.onJsonStateChange}
          />
        </div>
      </form>
    );
  }
}

ServiceForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  service: PropTypes.object.isRequired,
  onlySensorNameShards: PropTypes.bool.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
};

export default ServiceForm;
