import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import ShowActions from '../../components/ShowActions';
import LabeledValue from '../../components/LabeledValue';
import TableViewer from '../../components/TableViewer';
import Panel from '../../components/Panel';
import { ReadonlyRow as AggrRuleRow } from './AggrRulesTable';

import { isDefaultItemState } from '../../utils/itemState';
import ShardModalForm from './ShardModalForm';
import Modal from '../../components/Modal';
import { TabPane, Tabs } from '../../components/Tabs';
import EntityInfo from '../../components/EntityInfo';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';
import { isCloud } from '../../utils/env';
import AssociatedEntitiesTable from '../../components/AssociatedEntitiesTable';
import ConfirmDeletionModal from '../../components/ShowActions/ConfirmDeletionModal';

import { findService, deleteService } from '../../store/reducers/services/service';
import { findServiceClusters } from '../../store/reducers/services/serviceClusters';
import { findProjectClusters } from '../../store/reducers/clusters/clustersPage';
import {
  createShard, deleteShard, fetchShard, updateShard,
} from '../../api/shards';
import { PULL_PROTOCOL_SCHEME } from './constants';

// TODO: move to better place
const MM_PULL = 'PULL';
const MM_PUSH = 'PUSH';

const GenericOptions = ({ service, onlySensorNameShards, projectMetricNameLabel }) => {
  const model = ('port' in service && service.port !== 0) ? MM_PULL : MM_PUSH;
  const addTsArgs = service.addTsArgs || false;
  const rawDataMemOnly = (service.sensorConf && service.sensorConf.rawDataMemOnly) || false;

  let metricNameLabel = '';
  let metricNameLabelHint = '';
  if (service.sensorNameLabel) {
    metricNameLabel = service.sensorNameLabel;
  } else if (projectMetricNameLabel) {
    metricNameLabel = projectMetricNameLabel;
    metricNameLabelHint = 'defined in project';
  } else if (onlySensorNameShards) {
    metricNameLabel = isCloud() ? 'name' : 'sensor';
    metricNameLabelHint = 'default value for project with required metric name shards';
  } else {
    metricNameLabel = '';
  }

  let gridValue;
  if (service.gridSec) {
    gridValue = `${service.gridSec} seconds`;
  } else if (service.interval) {
    gridValue = `${service.interval} seconds (from interval)`;
  } else {
    gridValue = '0';
  }

  let protocolScheme;
  if (service.protocol && service.protocol in PULL_PROTOCOL_SCHEME) {
    protocolScheme = PULL_PROTOCOL_SCHEME[service.protocol];
  } else {
    protocolScheme = 'http';
  }

  return (
    <div>
      <LabeledValue label="Label name" value={service.name} />
      {metricNameLabel && (
        <LabeledValue
          label="Metric name label"
          value={metricNameLabel}
          hint={metricNameLabelHint}
        />
      )}
      <LabeledValue label="Monitoring model" value={model} />
      {model === MM_PULL
        && (
        <LabeledValue
          label="Fetch URL"
          value={`${protocolScheme}://{host}:${service.port || 80}${service.path}`}
        />
        )}
      {model === MM_PULL
        && (
        <LabeledValue
          label="Fetch interval"
          value={`${service.interval || 15} seconds`}
        />
        )}
      <LabeledValue label="Grid" value={gridValue} />
      {model === MM_PULL
        && <LabeledValue label="Add timestamp args" value={addTsArgs} />}
      <LabeledValue label="Store only aggregates" value={rawDataMemOnly} />
      {service.sensorsTtlDays
        ? <LabeledValue label="Metrics TTL" value={`${service.sensorsTtlDays} days`} />
        : null}
      {!!service.tvmDestId && (
        <LabeledValue
          label="TVM destination id"
          value={service.tvmDestId}
        />
      )}
      <EntityInfo entity={service} />
    </div>
  );
};

GenericOptions.propTypes = {
  service: PropTypes.object.isRequired,
  onlySensorNameShards: PropTypes.bool.isRequired,
  projectMetricNameLabel: PropTypes.string.isRequired,
};

class ServicePage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      newShardModalShown: false,
      deleteShardModalShown: false,
      errorMessage: '',
      askAboutShardDeletion: true,
      deletedShardId: '',
    };
  }

  componentDidMount() {
    const { projectId, serviceId } = this.props.match.params;
    this.props.findService(projectId, serviceId);
    this.props.findServiceClusters(projectId, serviceId);
    this.props.findProjectClusters(projectId, { pageSize: 'all' });
  }

  componentDidUpdate(prevProps) {
    const { projectId, serviceId } = this.props.match.params;
    const prevId = prevProps.match.params.serviceId;
    if (serviceId !== prevId) {
      this.props.findService(projectId, serviceId);
      this.props.findServiceClusters(projectId, serviceId);
      this.props.findProjectClusters(projectId, { pageSize: 'all' });
    }
  }

  onDelete = (event) => {
    event.preventDefault();
    const { match, history } = this.props;
    const { projectId, serviceId } = match.params;
    this.props.deleteService(projectId, serviceId)
      .then((action) => {
        if (action) {
          history.push(`/admin/projects/${projectId}/services`);
        }
      });
  };

  onAddNewShard = () => {
    /* eslint-disable react/no-string-refs */
    const state = this.refs.shardModalForm.stateToValue();
    const { projectId, serviceId } = this.props.match.params;
    return createShard(projectId, state)
      .catch((error) => this.setState({ errorMessage: error.message }))
      .then(this.onHideNewShardModal)
      .then(() => this.props.findServiceClusters(projectId, serviceId));
  };

  onDeactivateShard = (association) => {
    const { projectId, serviceId } = this.props.match.params;
    const { shardId } = association;
    fetchShard(projectId, shardId)
      .then((shard) => updateShard(projectId, { ...shard, state: 'INACTIVE' }))
      .then(() => this.props.findServiceClusters(projectId, serviceId));
  };

  onActivateShard = (association) => {
    const { projectId, serviceId } = this.props.match.params;
    const { shardId } = association;
    fetchShard(projectId, shardId)
      .then((shard) => updateShard(projectId, { ...shard, state: 'RW' }))
      .then(() => this.props.findServiceClusters(projectId, serviceId));
  };

  onDeleteShard = (shardId) => {
    const { projectId, serviceId } = this.props.match.params;
    return deleteShard(projectId, shardId).then(() => {
      this.props.findServiceClusters(projectId, serviceId);
    });
  };

  onShowNewShardModal = () => {
    this.setState({ newShardModalShown: true });
  };

  onHideNewShardModal = () => {
    this.setState({ newShardModalShown: false });
  };

  onConfirmDeletionOrDeleteShard = (association) => {
    const { shardId } = association;
    if (this.state.askAboutShardDeletion) {
      this.setState({ deleteShardModalShown: true, deletedShardId: shardId });
    } else {
      this.onDeleteShard(shardId);
    }
  };

  onConfirmShardDeletion = (event, dontAskAnymore) => {
    const shardId = this.state.deletedShardId;
    this.setState({ deleteShardModalShown: false, askAboutShardDeletion: dontAskAnymore, deletedShardId: '' }, () => {
      this.onDeleteShard(shardId);
    });
  };

  onHideDeleteShardModal = () => {
    this.setState({ deleteShardModalShown: false, deletedShardId: '' });
  };

  render() {
    const {
      service, serviceClusters, clustersPage,
      match, projectAuth,
      onlySensorNameShards, projectMetricNameLabel,
    } = this.props;
    const { projectId, serviceId } = match.params;

    if (isEmpty(service)) {
      return <span>Loading...</span>;
    }

    const activeServiceClusters = serviceClusters.filter((c) => isDefaultItemState(c.state));
    const inactiveServiceClusters = serviceClusters.filter((c) => !isDefaultItemState(c.state));

    const serviceClusterIds = serviceClusters.map((s) => s.id);
    const allClusterIds = (clustersPage.result || []).map((s) => s.id);
    const otherClusterIds = allClusterIds.filter((s) => serviceClusterIds.indexOf(s) < 0);

    let aggrRules = [];
    if (service.sensorConf) {
      aggrRules = service.sensorConf.aggrRules || [];
    }

    const canEdit = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDelete = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_DELETE);

    const canCreateShard = canEdit && otherClusterIds.length > 0;

    const newShardModal = (
      <Modal
        title="Add new shard"
        okTitle="Save"
        onOk={this.onAddNewShard}
        onCancel={this.onHideNewShardModal}
        isOpen={this.state.newShardModalShown}
      >
        <ShardModalForm
          ref="shardModalForm"
          projectId={projectId}
          serviceId={serviceId}
          clusters={otherClusterIds}
          errorMessage={this.state.errorMessage}
          onSubmit={this.onAddNewShard}
        />
      </Modal>
    );

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Service ${serviceId}`} />
        <div className="row">
          <div className="col-lg-6">
            <Panel title="Generic options">
              <GenericOptions
                service={service}
                onlySensorNameShards={onlySensorNameShards}
                projectMetricNameLabel={projectMetricNameLabel}
              />
            </Panel>
            <ShowActions
              editPath={`/admin/projects/${projectId}/services/${serviceId}/edit`}
              duplicatePath={`/admin/projects/${projectId}/services/new?duplicateOf=${serviceId}`}
              onDelete={this.onDelete}
              canEdit={canEdit}
              canChangeState={false}
              canDelete={canDelete && isEmpty(serviceClusters)}
            />
          </div>
          <div className="col-lg-6">
            {aggrRules.length > 0 && (
              <Panel title="Aggregation rules">
                <TableViewer
                  columns={['#', 'Conditions', 'Targets', 'Function', '']}
                  values={aggrRules}
                  row={AggrRuleRow}
                  limit={3}
                />
              </Panel>
            )}
            <Panel title="Associated clusters">
              <Tabs>
                <TabPane label="Active">
                  <div>
                    <AssociatedEntitiesTable
                      type="cluster"
                      projectId={projectId}
                      values={activeServiceClusters}
                      limit={5}
                      active
                      onAdd={canCreateShard ? this.onShowNewShardModal : null}
                      onToggle={canEdit ? this.onDeactivateShard : null}
                      onDelete={canDelete ? this.onConfirmDeletionOrDeleteShard : null}
                    />
                    {newShardModal}
                  </div>
                </TabPane>
                <TabPane label="Inactive">
                  <AssociatedEntitiesTable
                    type="cluster"
                    projectId={projectId}
                    values={inactiveServiceClusters}
                    limit={5}
                    active={false}
                    onToggle={canEdit ? this.onActivateShard : null}
                    onDelete={canDelete ? this.onConfirmDeletionOrDeleteShard : null}
                  />
                </TabPane>
              </Tabs>
              <ConfirmDeletionModal
                isOpen={this.state.deleteShardModalShown}
                onConfirm={this.onConfirmShardDeletion}
                onCancel={this.onHideDeleteShardModal}
              />
            </Panel>
          </div>
        </div>
      </div>
    );
  }
}

ServicePage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  service: PropTypes.object.isRequired,
  serviceClusters: PropTypes.array.isRequired,
  clustersPage: PropTypes.object.isRequired,
  onlySensorNameShards: PropTypes.bool.isRequired,
  projectMetricNameLabel: PropTypes.string.isRequired,
  findService: PropTypes.func.isRequired,
  findServiceClusters: PropTypes.func.isRequired,
  findProjectClusters: PropTypes.func.isRequired,
  deleteService: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  service: state.service,
  serviceClusters: state.serviceClusters || [],
  clustersPage: state.clustersPage,
  onlySensorNameShards: state.projectData.project.onlySensorNameShards,
  projectMetricNameLabel: state.projectData.project.metricNameLabel,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findService,
  deleteService,
  findServiceClusters,
  findProjectClusters,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ServicePage);

export default wrapSubProjectReadPage(connectedPage);
