import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import Pagination from '../../components/Pagination';
import TableToolbar from '../../components/TableToolbar';

import { findProjectServices } from '../../store/reducers/services/servicesPage';

import { formatSearch, parseSearch } from '../../utils/url';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';

const mapSearchToState = (search) => {
  const params = parseSearch(search);
  return {
    text: params.get('text') || '',
    page: params.get('page') || 0,
    pageSize: params.get('pageSize') || '',
  };
};

const mapStateToSearch = (state) => `?${formatSearch(state)}`;

const cleanState = (state) => {
  const params = {};
  const { text, page, pageSize } = state;
  if (text) {
    params.text = text;
  }
  if (page !== 0) {
    params.page = page;
  }
  if (pageSize !== '' && pageSize !== 30) {
    params.pageSize = pageSize;
  }
  return params;
};

class ServicesPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.props.findProjectServices(this._projectId, this.state);
  }

  onFilterChange = (event) => {
    const text = event.target.value;
    this.doReload({ ...this.state, text, page: 0 });
  };

  onActivePageChange = (activePage) => {
    this.doReload({ ...this.state, page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ ...this.state, pageSize, page: 0 });
  };

  doReload(newState) {
    this.setState(newState);
    const newCleanedState = cleanState(newState);
    this.props.history.replace(mapStateToSearch(newCleanedState));
    this.props.findProjectServices(this._projectId, newCleanedState);
  }

  render() {
    const { servicesPage } = this.props;
    const services = servicesPage.result || [];
    const pagination = servicesPage.page;
    const indexOffset = (pagination !== undefined)
      ? pagination.pageSize * pagination.current
      : 0;

    const canCreate = this.props.projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Services of ${this.props.projectName}`} />

        <TableToolbar
          createNewLink={canCreate ? `/admin/projects/${this._projectId}/services/new` : null}
          filter={this.state.text}
          onFilterChange={this.onFilterChange}
        />

        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>#</th>
              <th>Id</th>
              <th>Label name</th>
            </tr>
          </thead>
          <tbody>
            {services.map((s, i) => (
              <tr key={s.id}>
                <td>{i + indexOffset + 1}</td>
                <td><Link to={`/admin/projects/${this._projectId}/services/${s.id}`} tabIndex={i + 1}>{s.id}</Link></td>
                <td>{s.name}</td>
              </tr>
            ))}
          </tbody>
        </table>
        {pagination && (
          <Pagination
            activePage={pagination.current + 1}
            pageCount={pagination.pagesCount}
            pageSize={pagination.pageSize}
            totalCount={pagination.totalCount}
            onActivePageChange={this.onActivePageChange}
            onPageSizeChange={this.onPageSizeChange}
          />
        )}
      </div>
    );
  }
}

ServicesPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  projectAuth: PropTypes.object.isRequired,
  servicesPage: PropTypes.object.isRequired,
  findProjectServices: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  servicesPage: state.servicesPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findProjectServices,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ServicesPage);

export default wrapSubProjectReadPage(connectedPage);
