import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import Pagination from '../../components/Pagination';

import { findGlobalShards } from '../../store/reducers/shards/globalShardsPage';

import { formatSearch, parseSearch } from '../../utils/url';
import { parseItemState } from '../../utils/itemState';
import { ADMIN } from '../../auth/Roles';

import Auth from '../../auth/Auth';
import LinkWithName from './LinkWithName';
import ShardsToolbar from './ShardsToolbar';

const mapSearchToState = (search) => {
  const params = parseSearch(search);
  return {
    text: params.get('text') || '',
    page: params.get('page') || 0,
    pageSize: params.get('pageSize') || '',
    shardState: parseItemState(params.get('state')),
  };
};

const mapStateToSearch = (state) => `?${formatSearch(state)}`;

const cleanState = (state) => {
  const params = {};
  const {
    text, page, pageSize, shardState,
  } = state;
  if (text) {
    params.text = text;
  }
  if (page !== 0) {
    params.page = page;
  }
  if (pageSize !== '' && pageSize !== 30) {
    params.pageSize = pageSize;
  }
  if (shardState) {
    params.state = shardState;
  }
  return params;
};

class GlobalShardsPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.props.findGlobalShards(cleanState(this.state));
  }

  onFilterChange = (event) => {
    const text = event.target.value;
    this.doReload({ ...this.state, text, page: 0 });
  };

  onShardStateChange = (shardState) => {
    this.doReload({ ...this.state, shardState, page: 0 });
  };

  onActivePageChange = (activePage) => {
    this.doReload({ ...this.state, page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ ...this.state, pageSize, page: 0 });
  };

  doReload(newState) {
    this.setState(newState);
    const newCleanedState = cleanState(newState);
    this.props.history.replace(mapStateToSearch(newCleanedState));
    this.props.findGlobalShards(newCleanedState);
  }

  render() {
    if (!Auth.hasRole(ADMIN)) {
      return (
        <div className="alert alert-danger">
          You have no permissions to see this page
        </div>
      );
    }

    const { globalShardsPage } = this.props;
    const shards = globalShardsPage.result || [];
    const pagination = globalShardsPage.page;
    const indexOffset = (pagination !== undefined)
      ? pagination.pageSize * pagination.current
      : 0;

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title="Shards" />

        <ShardsToolbar
          filter={this.state.text}
          shardState={this.state.shardState}
          onFilterChange={this.onFilterChange}
          onShardStateChange={this.onShardStateChange}
        />

        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>#</th>
              <th>Id</th>
              <th>Project</th>
              <th>Cluster</th>
              <th>Service</th>
            </tr>
          </thead>
          <tbody>
            {shards.map((s, i) => (
              <tr key={s.id}>
                <td>{i + indexOffset + 1}</td>
                <td>
                  <Link to={`/admin/projects/${s.projectId}/shards/${s.id}`}>
                    {s.id}
                  </Link>
                </td>
                <td>
                  <Link to={`/admin/projects/${s.projectId}`}>{s.projectId}</Link>
                </td>
                <td>
                  <LinkWithName name={s.clusterName}>
                    <Link to={`/admin/projects/${s.projectId}/clusters/${s.clusterId}`}>
                      {s.clusterId}
                    </Link>
                  </LinkWithName>
                </td>
                <td>
                  <LinkWithName name={s.serviceName}>
                    <Link to={`/admin/projects/${s.projectId}/services/${s.serviceId}`}>
                      {s.serviceId}
                    </Link>
                  </LinkWithName>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
        {pagination && (
          <Pagination
            activePage={pagination.current + 1}
            pageCount={pagination.pagesCount}
            pageSize={pagination.pageSize}
            totalCount={pagination.totalCount}
            onActivePageChange={this.onActivePageChange}
            onPageSizeChange={this.onPageSizeChange}
          />
        )}
      </div>
    );
  }
}

GlobalShardsPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  globalShardsPage: PropTypes.object.isRequired,
  findGlobalShards: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({ globalShardsPage: state.globalShardsPage });

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findGlobalShards,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(GlobalShardsPage);
