/* eslint-disable jsx-a11y/no-noninteractive-element-interactions */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import Dropdown from 'react-bootstrap/es/Dropdown';
import MenuItem from 'react-bootstrap/es/MenuItem';

import BackLink from '../../components/BackLink';
import ConfirmDeletionModal from '../../components/ShowActions/ConfirmDeletionModal';
import {
  isDefaultItemState, ITEM_STATE, ITEM_STATE_TITLES, ITEM_STATES,
} from '../../utils/itemState';

class ShardActions extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { modalShown: false };
  }

  onShowConfirmModal = () => {
    this.setState({ modalShown: true });
  };

  onCancelModal = (event) => {
    event.preventDefault();
    this.setState({ modalShown: false });
  };

  onConfirmDeletion = (event) => {
    this.setState({ modalShown: false });
    this.props.onDelete(event);
  };

  render() {
    const {
      editPath, shardState, duplicatePath,
      canEdit, canChangeShardState, canDelete,
    } = this.props;

    if (!canEdit && !canChangeShardState && !canDelete) {
      return null;
    }

    let changeStateTitle;
    let changeStateAction;
    let style;

    const isActive = isDefaultItemState(shardState);

    if (isActive) {
      changeStateTitle = 'Deactivate';
      style = 'danger';
      changeStateAction = () => this.props.onChangeShardState(ITEM_STATE.INACTIVE);
    } else {
      changeStateTitle = 'Activate';
      style = 'warning';
      changeStateAction = () => this.props.onChangeShardState(ITEM_STATE.RW);
    }

    let rightButton = null;

    if (canChangeShardState) {
      rightButton = (
        <Dropdown id="shard-right-btn" dropup>
          <button
            type="button"
            className={`btn btn-outline btn-${style}`}
            onClick={changeStateAction}
          >
            {changeStateTitle}
          </button>
          <Dropdown.Toggle bsStyle={style} className="btn-outline" />
          <Dropdown.Menu>
            {ITEM_STATES.map((itemState, index) => {
              if (itemState === shardState) {
                return null;
              }

              return (
                <MenuItem key={itemState} onSelect={() => this.props.onChangeShardState(itemState)}>
                  Switch to
                  {' '}
                  {ITEM_STATE_TITLES[index]}
                </MenuItem>
              );
            })
              .filter((item) => item !== null)}
            {canDelete && (
              <MenuItem onSelect={this.onShowConfirmModal}>
                Delete
              </MenuItem>
            )}
          </Dropdown.Menu>
        </Dropdown>
      );
    } else if (canDelete) {
      rightButton = (
        <button
          type="button"
          className="btn btn-danger btn-outline"
          onClick={this.onShowConfirmModal}
        >
          Delete
        </button>
      );
    }

    let editButton = null;

    if (canEdit) {
      editButton = (
        <Link to={editPath} className="btn btn-outline btn-primary">Edit</Link>
      );
    }

    return (
      <div className="row show-actions">
        <hr />
        <div className="col-xs-offset-2 col-xs-10">
          {editButton}
          &nbsp;
          {canEdit
          && <Link to={duplicatePath} className="btn btn-outline btn-primary">Duplicate</Link>}
          &nbsp;
          <BackLink>Back</BackLink>
          <div className="pull-right">
            {rightButton}
          </div>
        </div>
        <ConfirmDeletionModal
          isOpen={this.state.modalShown}
          onConfirm={this.onConfirmDeletion}
          onCancel={this.onCancelModal}
        />
      </div>
    );
  }
}

ShardActions.propTypes = {
  editPath: PropTypes.string.isRequired,
  shardState: PropTypes.string.isRequired,
  duplicatePath: PropTypes.string.isRequired,
  onChangeShardState: PropTypes.func.isRequired,
  onDelete: PropTypes.func.isRequired,
  canEdit: PropTypes.bool.isRequired,
  canChangeShardState: PropTypes.bool.isRequired,
  canDelete: PropTypes.bool.isRequired,
};

export default ShardActions;
