import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import ShardForm from './ShardForm';

import { parseSearch } from '../../utils/url';
import { wrapSubProjectEditPage } from '../projects/wrapSubProjectPage';

import { findShard, saveShard, clearShard } from '../../store/reducers/shards/shard';
import { findProjectClusters } from '../../store/reducers/clusters/clustersPage';
import { findProjectServices } from '../../store/reducers/services/servicesPage';

class ShardEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._shardId = props.match.params.shardId;
    this._isNew = !this._shardId;

    if (this._isNew) {
      this._shardId = parseSearch(props.location.search).get('duplicateOf');
    }
  }

  componentDidMount() {
    const { shard } = this.props;
    if (this._shardId) {
      const loadedId = shard.id;
      if (loadedId !== this._shardId) {
        this.props.findShard(this._projectId, this._shardId);
      }
    } else {
      this.props.clearShard();
    }

    if (this._isNew) {
      this.props.findProjectClusters(this._projectId, { pageSize: 'all' });
      this.props.findProjectServices(this._projectId, { pageSize: 'all' });
    }
  }

  onSubmit = (value) => {
    const { history } = this.props;
    this.props.saveShard(this._projectId, value, this._isNew)
      .then(() => history.push(`/admin/projects/${this._projectId}/shards/${value.id}`));
  };

  render() {
    const {
      shard, clustersPage, servicesPage, onlySensorNameShards,
    } = this.props;
    const title = this._isNew ? 'New Shard' : `Shard ${this._shardId}`;
    const isLoading = (this._shardId && isEmpty(shard))
      || (this._isNew && (isEmpty(clustersPage.page) || isEmpty(servicesPage.page)));

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {isLoading && <span>Loading...</span>}
        {!isLoading
          && (
          <ShardForm
            projectId={this._projectId}
            shard={shard}
            clusters={clustersPage.result || []}
            services={servicesPage.result || []}
            onlySensorNameShards={onlySensorNameShards}
            isNew={this._isNew}
            onSubmit={this.onSubmit}
          />
          )}
      </div>
    );
  }
}

ShardEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  shard: PropTypes.object.isRequired,
  clustersPage: PropTypes.object.isRequired,
  servicesPage: PropTypes.object.isRequired,
  onlySensorNameShards: PropTypes.bool.isRequired,
  findShard: PropTypes.func.isRequired,
  findProjectClusters: PropTypes.func.isRequired,
  findProjectServices: PropTypes.func.isRequired,
  clearShard: PropTypes.func.isRequired,
  saveShard: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  shard: state.shard,
  clustersPage: state.clustersPage,
  servicesPage: state.servicesPage,
  onlySensorNameShards: state.projectData.project.onlySensorNameShards,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findShard,
  saveShard,
  clearShard,
  findProjectClusters,
  findProjectServices,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ShardEditPage);

export default wrapSubProjectEditPage(connectedPage);
