import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormButtons from '../../components/forms/FormButtons';
import FormInput from '../../components/forms/FormInput';
import ApiCallExample from '../../components/ApiCallExample';
import FormDatalist from '../../components/forms/FormDatalist';

import { inputTargetValue } from '../../utils/inputs';

import Auth from '../../auth/Auth';
import { ADMIN } from '../../auth/Roles';
import FormRadioGroup from '../graphs/FormRadioGroup';
import FormMetricNameLabel from '../../components/forms/FormMetricNameLabel';
import FormSelect from '../../components/forms/FormSelect';

function makeSuffix(prefix, value) {
  if (value.startsWith(prefix) && value !== prefix) {
    return value.substr(prefix.length);
  }
  return value;
}

export const makeShardId = (projectId, clusterId = '', serviceId = '') => {
  const shardPrefix = `${projectId}_`;
  const clusterSuffix = makeSuffix(shardPrefix, clusterId);
  const serviceSuffix = makeSuffix(shardPrefix, serviceId);
  if (clusterSuffix.length > 0 && serviceSuffix.length > 0) {
    return `${shardPrefix}${clusterSuffix}_${serviceSuffix}`;
  }
  return '';
};

export const VALIDATION_MODE_OPTIONS = [
  { value: 'legacySkip', title: 'Legacy skip' },
  { value: 'strictSkip', title: 'Strict skip' },
  { value: 'strictFail', title: 'Strict fail' },
];

export const DECIM_POLICY_OPTIONS = [
  { value: 'UNDEFINED', title: 'Default' },
  { value: 'POLICY_KEEP_FOREVER', title: 'Keep forever' },
  { value: 'POLICY_5_MIN_AFTER_7_DAYS', title: '5 min after 7 days' },
  { value: 'POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS', title: '1 min after month, 5 min after 3 months' },
  { value: 'POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS', title: '1 min after month, 5 min after 2 months' },
  { value: 'POLICY_5_MIN_AFTER_2_MONTHS', title: '5 min after 2 months' },
  { value: 'POLICY_5_MIN_AFTER_8_DAYS', title: '5 min after 8 days' },
];

class ShardForm extends PureComponent {
  static shardToState = (shard, projectId, isNew) => {
    if (shard.id) {
      const state = { projectId, ...shard };

      if (isNew) {
        state.id = makeShardId(projectId, state.clusterId, state.serviceId);
        delete state.hosts; // new shard must be balanced on diferent hosts
      }
      if (!Auth.hasRole(ADMIN)) {
        delete state.quotas;
        delete state.hosts;
        delete state.validationMode;
      }
      return state;
    }
    return { id: '', projectId };
  };

  constructor(props) {
    super(props);
    this.state = ShardForm.shardToState(props.shard, props.projectId, props.isNew);
    this._isNew = props.isNew;
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    // We cannot use setState() because it performs a shallow merge of
    // previous state with the new one. But here we need entirely update state.
    if (this.props.projectId !== prevProps.projectId || this.props.shard !== prevProps.shard) {
      this.state = ShardForm.shardToState(this.props.shard, this.props.projectId, false);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    this.setState({ [name]: value });
  };

  onSensorNameLabelChange = (sensorNameLabel) => {
    this.setState({ sensorNameLabel });
  };

  onClusterOrServiceChange = (event) => {
    const { name, value } = event.target;
    const clusterId = name === 'clusterId' ? value : this.state.clusterId;
    const serviceId = name === 'serviceId' ? value : this.state.serviceId;
    const id = makeShardId(this.props.projectId, clusterId, serviceId);
    this.setState({ id, clusterId, serviceId });
  };

  onQuotaChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    const quotas = { ...this.state.quotas };
    quotas[name] = value;
    this.setState({ quotas });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.state);
  };

  onJsonStateChange = (newState) => {
    this.setState(newState);
    this.forceUpdate();
  };

  render() {
    const { projectId, clusters, services } = this.props;
    const {
      id, clusterId, serviceId, sensorsTtlDays, validationMode, decimPolicy, sensorNameLabel,
    } = this.state;
    const quotas = this.state.quotas || {};

    const canUpdateInternals = Auth.hasRole(ADMIN);

    return (
      <form className="form-horizontal">
        <div className="col-lg-6 col-md-7">
          <FormInput
            type="text" name="id" label="Id" value={id}
            onChange={this.onInputChange}
            disabled={!this._isNew}
            help="Unique shard identifier. It is composed from project, cluster and service id automatically"
          />
          <FormDatalist
            type="text" name="clusterId" label="Cluster"
            value={clusterId} onChange={this.onClusterOrServiceChange}
            datalist={clusters.map((c) => c.id)}
            disabled={!this._isNew}
            help="Cluster unique identifier"
          />
          <FormDatalist
            type="text" name="serviceId" label="Service"
            value={serviceId} onChange={this.onClusterOrServiceChange}
            datalist={services.map((s) => s.id)}
            disabled={!this._isNew}
            help="Service unique identifier"
          />
          <FormMetricNameLabel
            type="shard"
            label="Metric name label"
            name="sensorNameLabel"
            value={sensorNameLabel}
            onChange={this.onSensorNameLabelChange}
            onlyMetricNameShards={this.props.onlySensorNameShards}
          />
          {canUpdateInternals && (
            <FormInput
              type="number" name="maxSensorsPerUrl" label="Max metrics per URL"
              value={quotas.maxSensorsPerUrl}
              onChange={this.onQuotaChange}
              help="This field defines how many metrics each host can send to Solomon.
                    Default value is 10 000. You can leave this field empty to use default value"
            />
          )}
          {canUpdateInternals && (
            <FormInput
              type="number" name="maxFileSensors" label="Max metrics"
              value={quotas.maxFileSensors}
              onChange={this.onQuotaChange}
              help="This field defines how many unique metrics can be stored in this shard.
                    Default value is 1 000 000. You can leave this field empty to use default value"
            />
          )}
          {canUpdateInternals && (
            <FormInput
              type="number" name="maxMemSensors" label="Max memonly metrics"
              value={quotas.maxMemSensors}
              onChange={this.onQuotaChange}
              help="This is field defines the limit of how many unique memonly metrics can be created for this shard.
                    Default value is 500 000. You can leave this field empty to use default value"
            />
          )}
          {canUpdateInternals && (
            <FormInput
              type="number" name="maxResponseSizeMb" label="Max response size (MB)"
              value={quotas.maxResponseSizeMb}
              onChange={this.onQuotaChange}
              help="This is field defines the maximum file size of sending metrics to Solomon.
                    Default value is 10 MB. You can leave this field empty to use default value"
            />
          )}
          <FormInput
            type="number"
            name="sensorsTtlDays"
            label="Metrics TTL"
            value={sensorsTtlDays}
            onChange={this.onInputChange}
            suffix="days"
            // eslint-disable-next-line max-len
            help="If this field is filled, Solomon will drop all metrics where last point was metrics TTL days ago"
          />
          {canUpdateInternals && (
            <FormRadioGroup
              name="validationMode"
              label="Validation mode"
              options={VALIDATION_MODE_OPTIONS}
              value={validationMode}
              defaultValue="legacySkip"
              onChange={this.onInputChange}
            />
          )}
          {canUpdateInternals && (
            <FormSelect
              name="decimPolicy"
              label="Decim policy"
              options={DECIM_POLICY_OPTIONS}
              value={decimPolicy}
              defaultValue="UNDEFINED"
              onChange={this.onInputChange}
            />
          )}
          <FormButtons onSubmit={this.onSubmit} />
        </div>

        <div className="col-lg-6 col-md-5">
          <ApiCallExample
            path={`/projects/${projectId}/shards`}
            value={this.state}
            isNew={this._isNew}
            onChange={this.onJsonStateChange}
          />
        </div>
      </form>
    );
  }
}

ShardForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  shard: PropTypes.object.isRequired,
  clusters: PropTypes.array.isRequired,
  services: PropTypes.array.isRequired,
  onlySensorNameShards: PropTypes.bool.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
};

export default ShardForm;
