import React, { PureComponent } from 'react';
import { Link } from 'react-router-dom';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import Panel from '../../components/Panel';
import LabeledValue from '../../components/LabeledValue';
import HostLabelWithRefresh from '../../components/HostLabel';
import Auth from '../../auth/Auth';
import { VALIDATION_MODE_OPTIONS } from './ShardForm';
import ShardActions from './ShardActions';

import { ITEM_STATE, parseItemState } from '../../utils/itemState';
import LinkWithName from './LinkWithName';
import EntityInfo from '../../components/EntityInfo';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { ADMIN } from '../../auth/Roles';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';
import { isCloud, isPreOrProdIntEnv } from '../../utils/env';
import Selectors from '../../utils/Selectors';
import Selector from '../../utils/Selector';
import { getMonitoringUiHost } from '../old/utils/graphToExpressionTransformations';

import {
  findShardWithRefs, activateShard, changeShardState, deleteShard, reloadShard,
} from '../../store/reducers/shards/shard';
import ShardQuotaField, { QUOTA_IDS } from './ShardQuotaField';
import { loadUserSettings } from '../../store/reducers/others/userSettings';

const ACTUAL_COREMON_PORT = 4510;
const ACTUAL_FETCHER_PORT = 4520;

const HostLinks = ({
  hosts, port, path, onRefresh,
}) => (
  <span>
    {hosts.map((h) => {
      if (!!port && !!path) {
        return (
          <a key={h} href={`${'/staffOnly/'}${h}:${port}${path}`}>
            <HostLabelWithRefresh host={h} onRefresh={onRefresh} />
          </a>
        );
      }

      return (
        <HostLabelWithRefresh host={h} onRefresh={onRefresh} />
      );
    })}
  </span>
);

HostLinks.propTypes = {
  hosts: PropTypes.arrayOf(PropTypes.string).isRequired,
  port: PropTypes.number,
  path: PropTypes.string,
  onRefresh: PropTypes.func.isRequired,
};

HostLinks.defaultProps = {
  port: 0,
  path: '',
};

function isText(value) {
  if (!value) {
    return false;
  }
  try {
    return JSON.parse(value).queryViewMode === 'text';
  } catch (e) {
    console.error(e);
    return false;
  }
}

function makeShardUiLink(shard) {
  if (shard.clusterName && shard.serviceName) {
    return (
      <a href={`/?project=${shard.projectId}&cluster=${shard.clusterName}&service=${shard.serviceName}`}>
        Shard graphs
      </a>
    );
  }
  return '-';
}

function makeShardMetricsSearch(shard) {
  try {
    const selectors = new Selectors(
      '',
      [
        Selector.glob('cluster', shard.clusterName),
        Selector.glob('service', shard.serviceName),
      ],
    );
    const selectorsStr = selectors.format();
    return `selectors=${encodeURIComponent(selectorsStr)}`;
  } catch (e) {
    console.error('failed to make shard metrics search', e);
    return '';
  }
}

function makeAdminShardMetricsLink(shard) {
  const search = makeShardMetricsSearch(shard);
  const pathname = `/admin/projects/${shard.projectId}/metrics`;

  return (
    <Link to={{ pathname, search }}>
      Shard metrics (in admin)
    </Link>
  );
}

function makeShardCardinalityAnalysisLink(shard) {
  return (
    <div>
      <a href={`https://datalens.yandex-team.ru/cwgldoijvgki5?projectId=${shard.projectId}&shardId=${shard.id}`} target="_blank" rel="noreferrer">
        Shard cardinality analysis dashboard (new!)
      </a>
    </div>
  );
}

function makeAdminShardStrictValidationLink(shard) {
  const search = makeShardMetricsSearch(shard);
  const pathname = `/admin/projects/${shard.projectId}/strictValidation`;

  return (
    <Link to={{ pathname, search }}>
      Check shard metrics for strict validation
    </Link>
  );
}

function findValidationModeTitle(value) {
  const option = VALIDATION_MODE_OPTIONS.filter((o) => o.value === value);
  return option.length > 0 ? option[0].title : 'Legacy skip';
}

export const decimPolicyTitles = {
  UNDEFINED: 'Default',
  POLICY_KEEP_FOREVER: 'Keep forever',
  POLICY_5_MIN_AFTER_7_DAYS: '5 min after 7 days',
  POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS: '1 min after month, 5 min after 3 months',
  POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS: '1 min after month, 5 min after 2 months',
  POLICY_5_MIN_AFTER_2_MONTHS: '5 min after 2 months',
  POLICY_5_MIN_AFTER_8_DAYS: '5 min after 8 days',
};

function findDecimPolicyTitle(value) {
  if (!value) {
    return 'Default';
  }
  return decimPolicyTitles[value] || value;
}

class ShardPage extends PureComponent {
  componentDidMount() {
    const { match } = this.props;
    this.props.findShardWithRefs(match.params.projectId, match.params.shardId);
    this.props.loadUserSettings();
  }

  componentDidUpdate(prevProps) {
    const { projectId, shardId } = this.props.match.params;
    const prevId = prevProps.match.params.shardId;
    if (shardId !== prevId) {
      this.props.findShardWithRefs(projectId, shardId);
    }
  }

  onChangeShardState = (state) => {
    const { shard } = this.props;
    this.props.changeShardState(shard, state);
  };

  onActivate = (event) => {
    event.preventDefault();
    const { shard } = this.props;
    this.props.activateShard(shard);
  };

  onDelete = (event) => {
    event.preventDefault();
    const { match, history } = this.props;
    const { projectId, shardId } = match.params;
    this.props.deleteShard(projectId, shardId)
      .then((action) => {
        if (action) {
          history.push(`/admin/projects/${projectId}/shards`);
        }
      });
  };

  onReload = (host, service) => {
    const { shard } = this.props;
    this.props.reloadShard(shard.projectId, shard.id, host, service);
  };

  render() {
    const {
      shard, service, cluster, match, projectAuth,
      onlySensorNameShards, projectMetricNameLabel,
    } = this.props;

    if (!match.params.shardId || isEmpty(shard) || isEmpty(service)) {
      return null;
    }

    const {
      id, numId, projectId, clusterId, serviceId, clusterName, serviceName,
    } = shard;

    const monitoringModel = ('port' in service && service.port !== 0) ? 'PULL' : 'PUSH';

    let metricNameLabel = '';
    let metricNameLabelHint = '';

    if (shard.sensorNameLabel) {
      metricNameLabel = shard.sensorNameLabel;
    } else if (service.sensorNameLabel) {
      metricNameLabel = service.sensorNameLabel;
      metricNameLabelHint = 'defined in service';
    } else if (projectMetricNameLabel) {
      metricNameLabel = projectMetricNameLabel;
      metricNameLabelHint = 'defined in project';
    } else if (onlySensorNameShards) {
      metricNameLabel = isCloud() ? 'name' : 'sensor';
      metricNameLabelHint = 'default value for project with required metric name shards';
    }

    let ttlLabelEl = null;
    if (shard.sensorsTtlDays) {
      ttlLabelEl = <LabeledValue label="Metrics TTL" value={`${shard.sensorsTtlDays} days`} />;
    } else if (cluster.sensorsTtlDays) {
      ttlLabelEl = <LabeledValue label="Metrics TTL" value={`${cluster.sensorsTtlDays} days`} hint="defined in cluster" />;
    } else if (service.sensorsTtlDays) {
      ttlLabelEl = <LabeledValue label="Metrics TTL" value={`${service.sensorsTtlDays} days`} hint="defined in service" />;
    }

    const projectLink = (<Link to={`/admin/projects/${projectId}`}>{projectId}</Link>);
    const clusterLink = (
      <LinkWithName name={clusterName}>
        <Link to={`/admin/projects/${projectId}/clusters/${clusterId}`}>{clusterId}</Link>
      </LinkWithName>
    );
    const serviceLink = (
      <LinkWithName name={serviceName}>
        <Link to={`/admin/projects/${projectId}/services/${serviceId}`}>{serviceId}</Link>
      </LinkWithName>
    );

    const shardQuotas = shard.quotas || {};
    const hosts = shard.hosts || [];

    const canEdit = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDelete = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_DELETE);

    const shardState = parseItemState(shard.state);

    let subTitle;

    switch (shardState) {
      case ITEM_STATE.READ_ONLY:
        subTitle = (<span className="label label-info">read only</span>);
        break;
      case ITEM_STATE.WRITE_ONLY:
        subTitle = (<span className="label label-info">write only</span>);
        break;
      case ITEM_STATE.INACTIVE:
        subTitle = (<span className="label label-warning">inactive</span>);
        break;
      default:
        subTitle = null;
    }

    const type = isText(this.props.uiSettings) ? 'text' : 's';
    // Грязный обход всех прелюдий
    const explorerUrl = `https://${getMonitoringUiHost()}/projects/${projectId}/explorer/queries?q.0.${type}={project="${projectId}",cluster="${clusterName}",service="${serviceName}"}&utm_source=solomon_shard_graphs`;
    const explorerLink = (
      <a href={explorerUrl}>
        <strong>Shard graphs in Monitoring. New!</strong>
      </a>
    );

    // eslint-disable-next-line no-bitwise
    const unsignedNumId = numId >>> 0;
    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Shard ${id}`} subTitle={subTitle} />

        <div className="row">
          <div className="col-xs-6">
            <Panel title="Generic options">
              <LabeledValue label="Id" value={id} />
              {Auth.hasRole(ADMIN) && <LabeledValue label="NumId" value={unsignedNumId} />}
              <LabeledValue label="Project" value={projectLink} />
              <LabeledValue label="Cluster" value={clusterLink} />
              <LabeledValue label="Service" value={serviceLink} />
              <LabeledValue label="Monitoring model" value={monitoringModel} />
              {metricNameLabel && (
                <LabeledValue
                  label="Metric name label"
                  value={metricNameLabel}
                  hint={metricNameLabelHint}
                />
              )}
              {ttlLabelEl}
              <ShardQuotaField
                quotaId={QUOTA_IDS.MAX_SENSORS_PER_URL_FORM_ID}
                projectId={projectId}
                shardId={id}
                label="Max metrics per URL"
                value={shardQuotas.maxSensorsPerUrl}
                formatted
              />
              <ShardQuotaField
                quotaId={QUOTA_IDS.MAX_FILE_SENSORS_FORM_ID}
                projectId={projectId}
                shardId={id}
                label="Max metrics"
                value={shardQuotas.maxFileSensors}
                formatted
              />
              <ShardQuotaField
                quotaId={QUOTA_IDS.MAX_MEMONLY_SENSORS_FORM_ID}
                projectId={projectId}
                shardId={id}
                label="Max memonly metrics"
                value={shardQuotas.maxMemSensors}
                formatted
              />
              <ShardQuotaField
                quotaId={QUOTA_IDS.MAX_RESPONSE_SIZE_FORM_ID}
                projectId={projectId}
                shardId={id}
                label="Max response size (MB)"
                value={shardQuotas.maxResponseSizeMb}
                formatted={false}
              />
              <LabeledValue label="Validation mode" value={findValidationModeTitle(shard.validationMode)} />
              <LabeledValue label="Decim policy" value={findDecimPolicyTitle(shard.decimPolicy)} />
              <EntityInfo entity={shard} />
              {isPreOrProdIntEnv() && <LabeledValue value={explorerLink} />}
              <LabeledValue value={makeShardUiLink(shard)} />
              <LabeledValue value={makeAdminShardMetricsLink(shard)} />
              <LabeledValue value={makeShardCardinalityAnalysisLink(shard)} />
              <LabeledValue value={makeAdminShardStrictValidationLink(shard)} />
              <LabeledValue value={(
                <Link to={`/admin/projects/${projectId}/shards/${id}/status`}>
                  Shard Status
                </Link>
)}
              />
            </Panel>
            <ShardActions
              editPath={`/admin/projects/${projectId}/shards/${id}/edit`}
              duplicatePath={`/admin/projects/${projectId}/shards/new?duplicateOf=${id}`}
              onChangeShardState={this.onChangeShardState}
              onDelete={this.onDelete}
              shardState={shardState}
              canEdit={canEdit}
              canChangeShardState={canEdit}
              canDelete={canDelete}
            />
          </div>
          <div className="col-xs-6">
            {Auth.hasRole(ADMIN)
              && (
              <Panel title="Shard internals on backends">
                <LabeledValue
                  label="Fetchers"
                  value={(
                    <HostLinks
                      hosts={hosts}
                      port={ACTUAL_FETCHER_PORT}
                      path={`/selfmon/shards?shardId=${shard.id}`}
                      onRefresh={(host) => this.onReload(host, 'fetcher')}
                    />
                  )}
                />
                <LabeledValue
                  label="Coremons"
                  value={(
                    <HostLinks
                      hosts={hosts}
                      port={ACTUAL_COREMON_PORT}
                      path={`/manager?bean=${shard.id}&class=ru.yandex.solomon.coremon.CoremonShard`}
                      onRefresh={(host) => this.onReload(host, 'coremon')}
                    />
)}
                />
              </Panel>
              )}
          </div>
        </div>
      </div>
    );
  }
}

ShardPage.propTypes = {
  match: PropTypes.object.isRequired,
  shard: PropTypes.object.isRequired,
  service: PropTypes.object.isRequired,
  cluster: PropTypes.object.isRequired,
  onlySensorNameShards: PropTypes.bool.isRequired,
  projectMetricNameLabel: PropTypes.string.isRequired,
  history: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  findShardWithRefs: PropTypes.func.isRequired,
  changeShardState: PropTypes.func.isRequired,
  activateShard: PropTypes.func.isRequired,
  deleteShard: PropTypes.func.isRequired,
  reloadShard: PropTypes.func.isRequired,
  loadUserSettings: PropTypes.func.isRequired,
  uiSettings: PropTypes.string,
};

ShardPage.defaultProps = {
  uiSettings: '',
};

const mapStateToProps = (state) => ({
  shard: state.shard,
  service: state.service,
  cluster: state.cluster,
  onlySensorNameShards: state.projectData.project.onlySensorNameShards,
  projectMetricNameLabel: state.projectData.project.metricNameLabel,
  uiSettings: state.userSettings.data.settings.ui,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findShardWithRefs,
  changeShardState,
  activateShard,
  deleteShard,
  reloadShard,
  loadUserSettings,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ShardPage);

export default wrapSubProjectReadPage(connectedPage);
