import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import RadioButtons from '../../../components/RadioButtons';
import { shortHostname } from '../../../utils/hosts';
import LocalDate from '../../../components/LocalDate';
import { formatDuration } from '../../../utils/duration';
import { numberToDisplaySize } from '../../../utils/number';
import Pagination from '../../../components/Pagination';
import StatusLabel from './StatusLabel';
import { fetchShardTargetsStatus, fetchServiceProviderTargetsStatus } from '../../../api/shards';
import WarningAlert from '../../projects/alerts/WarningAlert';
import { formatSearch, parseSearch } from '../../../utils/url';
import ShardStatusesModal from './ShardStatusesModal';

const FAIL_STATUSES = [
  'AUTH_ERROR',
  'QUOTA_ERROR',
  'RESPONSE_TOO_LARGE',
  'CONNECT_FAILURE',
  'TIMEOUT',
  'JSON_ERROR',
  'SPACK_ERROR',
  'NOT_200',
  'UNKNOWN_DC',
  'UNKNOWN_HOST',
  'UNKNOWN_SHARD',
  'IPC_QUEUE_OVERFLOW',
  'SHARD_NOT_INITIALIZED',
  'SKIP_TOO_LONG',
  'DERIV_AND_TS',
  'FETCHER_OOM',
  'SENSOR_OVERFLOW',
  'UNKNOWN_ERROR',
];

const DC_NAMES = [
  'MAN',
  'SAS',
  'MYT',
  'VLA',
  'FOL',
  'UGR',
  'IVA',
  'ASH',
  'AMS',
  'VEG',
];

const mapSearchToState = (search) => {
  const parsedSearch = parseSearch(search);
  return {
    fetcherHost: parsedSearch.get('fetcherHost') || '',
    page: parsedSearch.get('page') || 0,
    pageSize: parsedSearch.get('pageSize') || '',
    hostGlob: parsedSearch.get('hostGlob') || '',
    dc: parsedSearch.get('dc') || '',
    status: parsedSearch.get('status') || '',
  };
};

const mapStateToSearch = (state) => `?${formatSearch(state)}`;

const cleanState = (state) => {
  const params = {};
  const {
    fetcherHost, page, pageSize, hostGlob, dc, status,
  } = state;
  if (fetcherHost) {
    params.fetcherHost = fetcherHost;
  }
  if (page !== 0) {
    params.page = page;
  }
  if (pageSize !== '' && pageSize !== 30) {
    params.pageSize = pageSize;
  }
  if (hostGlob) {
    params.hostGlob = hostGlob;
  }
  if (dc) {
    params.dc = dc;
  }
  if (status) {
    params.status = status;
  }
  return params;
};

function transformLastContentType(lastContentType) {
  if (!lastContentType || lastContentType === 'UNKNOWN_CONTENT_TYPE') {
    return '-';
  }

  return lastContentType;
}

function writeStatusButtonText(ss) {
  const errCnt = ss.reduce((cnt, s) => cnt + (!!s.lastError || (s.lastStatus in FAIL_STATUSES)), 0);
  return errCnt ? `${errCnt} errors` : 'Show';
}

class FetchTargetsView extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      loading: false,
      error: null,
      shardTargetsStatusPage: null,
    };

    this.onClearShardStatuses_ = this.onClearShardStatuses.bind(this);
  }

  componentDidMount() {
    this.fetchData();
  }

  componentDidUpdate(prevProps) {
    if (prevProps.search !== this.props.search) {
      this.fetchData();
    }
  }

  onHostnameChange = (event) => {
    event.preventDefault();
    const hostGlob = event.target.value;
    this.doReload({ hostGlob, page: 0 });
  };

  onDcChange = (event) => {
    event.preventDefault();
    const dc = event.target.value;
    this.doReload({ dc, page: 0 });
  };

  onStatusChange = (event) => {
    event.preventDefault();
    const status = event.target.value;
    this.doReload({ status, page: 0 });
  };

  onActivePageChange = (activePage) => {
    this.doReload({ page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ pageSize, page: 0 });
  };

  onFetcherHostChange = (fetcherHost) => {
    this.doReload({ fetcherHost, page: 0 });
  };

  onClearShardStatuses() {
    this.setState({
      ...this.state,
      shardStatusesUrl: null,
      shardStatuses: null,
    });
  }

  onShowShardStatuses(url, shardStatuses) {
    this.setState({
      ...this.state,
      shardStatusesUrl: url,
      shardStatuses,
    });
  }

  doReload(newState) {
    const state = mapSearchToState(this.props.search);
    const mergedState = { ...state, ...newState };
    const cleanedState = cleanState(mergedState);
    const newSearch = mapStateToSearch(cleanedState);
    this.props.history.replace(newSearch);
  }

  async fetchData() {
    const { projectId, shardId, serviceProviderId } = this.props;

    const params = cleanState(mapSearchToState(this.props.search));

    this.setState({ loading: true, error: null });

    try {
      const shardTargetsStatusPage = serviceProviderId
        ? await fetchServiceProviderTargetsStatus(serviceProviderId, params)
        : await fetchShardTargetsStatus(projectId, shardId, params);

      const targets = shardTargetsStatusPage.result;

      if (!serviceProviderId) {
        targets.forEach((target, i) => {
          if (target.shardStatuses.length > 1) {
            console.error('a simple (not multishard) url has statuses for more than one shard');
          } else if (target.shardStatuses.length === 1) {
            // a shortcut for simple shards to keep the same behavior as before multishard urls
            targets[i].lastUrlStatus = target.shardStatuses[0].lastStatus;
            targets[i].lastUrlError = target.shardStatuses[0].lastError;
            targets[i].lastSensorsParsed = target.shardStatuses[0].lastSensorsParsed;
          }
        });
      } else {
        targets.forEach((target) => {
          target.shardStatuses.sort((left, right) => {
            const compBy = (fn) => fn(right) - fn(left);

            // errors first
            let comp = compBy((el) => !!el.lastError);
            if (comp) return comp;
            comp = compBy((el) => el.lastStatus in FAIL_STATUSES);
            if (comp) return comp;

            return 0;
          });
        });
      }

      this.setState({
        loading: false,
        error: null,
        shardTargetsStatusPage,
      });
    } catch (e) {
      console.error(e.message);
      this.setState({ loading: false, error: e.message, shardTargetsStatusPage: null });
    }
  }

  render() {
    const { shardHosts, serviceProviderId } = this.props;

    const params = mapSearchToState(this.props.search);
    const { hostGlob, dc, status } = params;
    const fetcherHost = params.fetcherHost || (shardHosts.length > 0 ? shardHosts[0] : '');

    const { loading, error, shardTargetsStatusPage } = this.state;

    let table = null;

    if (error) {
      table = (
        <WarningAlert title="Loading error" message={error} />
      );
    } else if (shardTargetsStatusPage) {
      const { result: targets, page: pagination } = shardTargetsStatusPage;

      const indexOffset = (pagination !== undefined)
        ? pagination.pageSize * pagination.current
        : 0;

      table = (
        <>
          <table className="table table-condensed table-hover">
            <thead>
              <tr>
                <th>#</th>
                <th>Hostname</th>
                <th>DC</th>
                <th>
                  {serviceProviderId ? 'Fetch Status' : 'Status'}
                  &nbsp;
                  <a
                    href="https://wiki.yandex-team.ru/solomon/userguide/FAQ/#error-types"
                    target="_blank"
                    rel="noopener noreferrer"
                  >
                    <i className="glyphicon glyphicon-info-sign" />
                  </a>
                </th>
                {serviceProviderId ? (<th>Write status</th>) : null}
                <th>Content type</th>
                <th>Fetch Time</th>
                <th>Fetch duration</th>
                <th>Response Bytes</th>
                <th>Metrics Parsed</th>
                <th>Error</th>
              </tr>
            </thead>
            <tbody>
              {targets.map((t, i) => (
                // eslint-disable-next-line react/no-array-index-key
                <tr key={i}>
                  <td>{i + indexOffset + 1}</td>
                  <td><a href={t.url}>{t.host}</a></td>
                  <td>{t.dc}</td>
                  <td><StatusLabel status={t.lastUrlStatus} /></td>
                  {serviceProviderId
                    ? (
                      <td>
                        <button type="button" className="btn btn-default" onClick={() => this.onShowShardStatuses(t.url, t.shardStatuses)}>
                          {writeStatusButtonText(t.shardStatuses)}
                        </button>
                      </td>
                    )
                    : null}
                  <td>{transformLastContentType(t.lastContentType)}</td>
                  <td><LocalDate date={t.lastFetchTime} /></td>
                  <td>{formatDuration(t.lastFetchDurationMillis || 0)}</td>
                  <td>{numberToDisplaySize(t.lastResponseBytes)}</td>
                  <td>{t.lastSensorsParsed}</td>
                  <td>{t.lastUrlError}</td>
                </tr>
              ))}
            </tbody>
          </table>
          {pagination && (
            <Pagination
              activePage={pagination.current + 1}
              pageCount={pagination.pagesCount}
              pageSize={pagination.pageSize}
              totalCount={pagination.totalCount}
              onActivePageChange={this.onActivePageChange}
              onPageSizeChange={this.onPageSizeChange}
            />
          )}
          {!!serviceProviderId && (
            <ShardStatusesModal
              url={this.state.shardStatusesUrl || ''}
              shardStatuses={this.state.shardStatuses || []}
              onClearShardStatuses={this.onClearShardStatuses_}
            />
          )}
        </>
      );
    } else if (loading) {
      table = <div>Loading...</div>;
    }

    return (
      <div>
        <div className="btn-toolbar table-toolbar">
          <div className="btn-group">
            <input
              type="search"
              className="form-control"
              placeholder="Hostname..."
              defaultValue={hostGlob}
              onChange={this.onHostnameChange}
              ref={(input) => { this.hostnameInput = input; }}
              tabIndex={0}
            />
          </div>
          <div className="btn-group">
            <select className="form-control" onChange={this.onDcChange} value={dc}>
              <option value="">Any DC</option>
              {DC_NAMES.map((s) => <option key={s} value={s}>{s}</option>)}
            </select>
          </div>
          <div className="btn-group">
            <select className="form-control" onChange={this.onStatusChange} value={status}>
              <option value="">Any status</option>
              <optgroup label="Success">
                <option>OK</option>
              </optgroup>
              <optgroup label="Failures">
                <option value="NOT_OK">Any error</option>
                {FAIL_STATUSES.map((s) => <option key={s} value={s}>{s}</option>)}
              </optgroup>
            </select>
          </div>
          <div className="pull-right">
            <RadioButtons
              choices={shardHosts}
              labels={shardHosts.map(shortHostname)}
              selected={fetcherHost}
              onSelect={this.onFetcherHostChange}
            />
          </div>
        </div>
        {table}
      </div>
    );
  }
}

FetchTargetsView.propTypes = {
  projectId: PropTypes.string.isRequired,
  shardId: PropTypes.string.isRequired,
  serviceProviderId: PropTypes.string.isRequired,
  shardHosts: PropTypes.array.isRequired,
  search: PropTypes.string.isRequired,
  history: PropTypes.object.isRequired,
};

export default FetchTargetsView;
