/* eslint-disable no-nested-ternary,no-restricted-syntax */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import Breadcrumb from '../../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../../components/PageTitle';
import { TabPane } from '../../../components/Tabs';
import TabsWithUrl from '../../../components/TabsWithUrl';
import { getUrlHash, setUrlHash } from '../../../utils/url';
import { wrapSubProjectReadPage } from '../../projects/wrapSubProjectPage';
import * as API from '../../../api/shards';
import { fetchService } from '../../../api/services';
import ProcessingGraphsView from './ProcessingGraphsView';
import PushGraphsView from './PushGraphsView';
import MetricsGraphsView from './MetricsGraphsView';
import FetchingGraphsView from './FetchingGraphsView';
import FetchTargetsView from './FetchTargetsView';
import WarningAlert from '../../projects/alerts/WarningAlert';
import StorageGraphsView from './StorageGraphsView';
import { hasProcessingShardDataForEnv, hasStorageShardDataForEnv, isPreOrProdIntEnv } from '../../../utils/env';

class ShardStatusPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      loading: false,
      error: null,
      id: '',
      shardHosts: [],
      shardNumId: '0',
      monitoringModel: 'UNKNOWN',
      duration: '1h',
      viewNewVersion: isPreOrProdIntEnv(),
    };
  }

  componentDidMount() {
    this.fetchData();
  }

  onDurationChange = (duration) => {
    this.setState({ duration });
  };

  onToggleViewNew = () => {
    this.setState({
      viewNewVersion: !this.state.viewNewVersion,
    });
  }

  async fetchData() {
    const { projectId, shardId } = this.props.match.params;

    this.setState({ loading: true, id: shardId, error: null });

    try {
      const shard = await API.fetchShard(projectId, shardId);
      const service = await fetchService(projectId, shard.serviceId);

      const monitoringModel = ('port' in service && service.port !== 0) ? 'PULL' : 'PUSH';

      // eslint-disable-next-line no-bitwise
      const unsignedNumId = shard.numId >>> 0;

      this.setState({
        loading: false,
        error: null,
        shardHosts: shard.hosts || [],
        shardNumId: `${unsignedNumId}`,
        monitoringModel,
      });
    } catch (error) {
      console.error(error);
      this.setState({
        loading: false,
        error: error.message,
      });
    }
  }

  render() {
    const { projectId, shardId } = this.props.match.params;

    const {
      loading,
      error,
      id,
      monitoringModel,
      shardHosts,
      shardNumId,
      duration,
      viewNewVersion,
    } = this.state;

    let content = null;

    if (loading) {
      content = <div>Loading...</div>;
    } else if (error) {
      content = (
        <WarningAlert title={error} />
      );
    } else if (id) {
      const isPull = monitoringModel === 'PULL';

      content = (
        <TabsWithUrl activeTabId={getUrlHash()} setActiveTab={setUrlHash}>
          {isPull && (
            <TabPane label="Targets">
              <FetchTargetsView
                projectId={projectId}
                shardId={shardId}
                shardHosts={shardHosts}
                search={this.props.location.search}
                history={this.props.history}
              />
            </TabPane>
          )}
          {isPull && (
            <TabPane label="Fetching">
              <FetchingGraphsView
                projectId={projectId}
                shardId={shardId}
                duration={duration}
                viewNewVersion={viewNewVersion}
                onToggleViewNew={this.onToggleViewNew}
                onDurationChange={this.onDurationChange}
              />
            </TabPane>
          )}
          <TabPane label="Push">
            <PushGraphsView
              projectId={projectId}
              shardId={shardId}
              duration={duration}
              viewNewVersion={viewNewVersion}
              onToggleViewNew={this.onToggleViewNew}
              onDurationChange={this.onDurationChange}
            />
          </TabPane>
          {hasProcessingShardDataForEnv && (
            <TabPane label="Processing">
              <ProcessingGraphsView
                projectId={projectId}
                shardId={shardId}
                duration={duration}
                viewNewVersion={viewNewVersion}
                onToggleViewNew={this.onToggleViewNew}
                onDurationChange={this.onDurationChange}
              />
            </TabPane>
          )}
          {hasStorageShardDataForEnv && (
            <TabPane label="Storage">
              <StorageGraphsView
                projectId={projectId}
                shardNumId={shardNumId}
                duration={duration}
                viewNewVersion={viewNewVersion}
                onToggleViewNew={this.onToggleViewNew}
                onDurationChange={this.onDurationChange}
              />
            </TabPane>
          )}
          <TabPane label="Metric limits">
            <MetricsGraphsView
              projectId={projectId}
              shardId={shardId}
              duration={duration}
              viewNewVersion={viewNewVersion}
              onToggleViewNew={this.onToggleViewNew}
              onDurationChange={this.onDurationChange}
            />
          </TabPane>
        </TabsWithUrl>
      );
    }

    const title = `Status of shard ${shardId}`;

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {content}
      </div>
    );
  }
}

ShardStatusPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
};

export default wrapSubProjectReadPage(ShardStatusPage);
