import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import Pagination from '../../components/Pagination';

import { formatSearch, parseSearch } from '../../utils/url';
import { isDefaultItemState, parseItemState } from '../../utils/itemState';
import LinkWithName from './LinkWithName';
import ShardsToolbar from './ShardsToolbar';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';

import { findProjectShards } from '../../store/reducers/shards/shardsPage';

const mapSearchToState = (search) => {
  const params = parseSearch(search);
  return {
    text: params.get('text') || '',
    page: params.get('page') || 0,
    pageSize: params.get('pageSize') || '',
    shardState: parseItemState(params.get('state')),
  };
};

const mapStateToSearch = (state) => `?${formatSearch(state)}`;

const cleanState = (state) => {
  const params = {};
  const {
    text, page, pageSize, shardState,
  } = state;
  if (text) {
    params.text = text;
  }
  if (page !== 0) {
    params.page = page;
  }
  if (pageSize !== '' && pageSize !== 30) {
    params.pageSize = pageSize;
  }
  if (shardState && !isDefaultItemState(shardState)) {
    params.state = shardState;
  }
  return params;
};

class ShardsPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.props.findProjectShards(this._projectId, cleanState(this.state));
  }

  onFilterChange = (event) => {
    const text = event.target.value;
    this.doReload({ ...this.state, text, page: 0 });
  };

  onShardStateChange = (shardState) => {
    this.doReload({ ...this.state, shardState, page: 0 });
  };

  onActivePageChange = (activePage) => {
    this.doReload({ ...this.state, page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ ...this.state, pageSize, page: 0 });
  };

  doReload(newState) {
    this.setState(newState);
    const newCleanedState = cleanState(newState);
    this.props.history.replace(mapStateToSearch(newCleanedState));
    this.props.findProjectShards(this._projectId, newCleanedState);
  }

  render() {
    const { shardsPage } = this.props;
    const shards = shardsPage.result || [];
    const pagination = shardsPage.page;
    const indexOffset = (pagination !== undefined)
      ? pagination.pageSize * pagination.current
      : 0;

    const canCreate = this.props.projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Shards of ${this.props.projectName}`} />

        <ShardsToolbar
          filter={this.state.text}
          createNewLink={canCreate ? `/admin/projects/${this._projectId}/shards/new` : null}
          shardState={this.state.shardState}
          onFilterChange={this.onFilterChange}
          onShardStateChange={this.onShardStateChange}
        />

        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>#</th>
              <th>Id</th>
              <th>Cluster</th>
              <th>Service</th>
            </tr>
          </thead>
          <tbody>
            {shards.map((s, i) => (
              <tr key={s.id}>
                <td>{i + indexOffset + 1}</td>
                <td><Link to={`/admin/projects/${this._projectId}/shards/${s.id}`}>{s.id}</Link></td>
                <td>
                  <LinkWithName name={s.clusterName}>
                    <Link to={`/admin/projects/${this._projectId}/clusters/${s.clusterId}`}>{s.clusterId}</Link>
                  </LinkWithName>
                </td>
                <td>
                  <LinkWithName name={s.serviceName}>
                    <Link to={`/admin/projects/${this._projectId}/services/${s.serviceId}`}>{s.serviceId}</Link>
                  </LinkWithName>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
        {pagination && (
          <Pagination
            activePage={pagination.current + 1}
            pageCount={pagination.pagesCount}
            pageSize={pagination.pageSize}
            totalCount={pagination.totalCount}
            onActivePageChange={this.onActivePageChange}
            onPageSizeChange={this.onPageSizeChange}
          />
        )}
      </div>
    );
  }
}

ShardsPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  shardsPage: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  findProjectShards: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  shardsPage: state.shardsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findProjectShards,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ShardsPage);

export default wrapSubProjectReadPage(connectedPage);
