import SvgUtils from '../svg/SvgUtils';
import ChartCommon from '../common/ChartCommon';
import YaxisCommon from '../common/YaxisCommon';
import Mathx from '../../utils/Mathx';
import UserLinksBasic from '../../utils/UserLinksBasic';

class ChartBars {
  /**
   * @param {ChartBarsCommon} chartBars
   * @param {GraphFormatter} graphFormatter
   * @param {AggrValue[]} values
   */
  static drawBars(chartBars, graphFormatter, values) {
    const { $chartArea } = chartBars.commonChartElements;

    const yZeroPos = Math.trunc(chartBars.commonChartElements.geom.dataYToChartAreaY('left', 0));
    for (let i = 0; i < values.length; ++i) {
      const aggrValue = values[i];

      const yPos = chartBars.valueToChartAreaY(aggrValue.value);

      const $rect = SvgUtils.createRectElement();
      $rect.setAttribute('fill', aggrValue.color);

      let x = chartBars.barsGeom.getBarLeft(i);
      let y = yZeroPos;
      let width = chartBars.barsGeom.getBarWidth();
      let height = yPos - yZeroPos;

      if (width < 0) {
        width = Mathx.absSat(width);
        x = Mathx.subSat(x, width);
      }
      if (height < 0) {
        height = Mathx.absSat(height);
        y = Mathx.subSat(y, height);
      }

      $rect.setAttribute('x', `${x}`);
      $rect.setAttribute('y', `${y}`);
      $rect.setAttribute('width', `${width}`);
      $rect.setAttribute('height', `${height}`);

      $chartArea.appendChild($rect);
    }

    $chartArea.addEventListener('mousemove', (event) => {
      const barIndex = ChartBars.barIndexFromCursorPosition(event, chartBars);
      if (barIndex >= 0 && barIndex < values.length) {
        const value = values[barIndex];
        chartBars.drawHoverInBarLeft(graphFormatter, value, barIndex);
      } else {
        chartBars.killHover();
      }
    });

    $chartArea.addEventListener('mousedown', (event) => {
      const barIndex = ChartBars.barIndexFromCursorPosition(event, chartBars);
      if (barIndex >= 0 && barIndex < values.length) {
        const value = values[barIndex];
        const { link } = value;
        if (link) {
          window.top.location.href = UserLinksBasic.fixOldAdminUrl(link);
        }
      }
    });

    $chartArea.addEventListener('mouseleave', () => {
      chartBars.killHover();
    });
  }

  static barIndexFromCursorPosition(mouseEvent, chartBars) {
    const cursorX = chartBars.cursorX(mouseEvent);
    return chartBars.barsGeom.getBarIndexFromCursorX(cursorX);
  }

  /**
   *
   * @param {ChartBarsCommon} chartBars
   * @param {GraphFormatter} graphFormatter
   * @param {AggrValue[]} values
   * @param {ChartConf} conf
   */
  static chart(chartBars, graphFormatter, values, conf) {
    const common = chartBars.commonChartElements;

    YaxisCommon.drawYAxisLabels(conf, common);

    YaxisCommon.drawGridY(common.geom, common.$chartArea);

    ChartBars.drawBars(chartBars, graphFormatter, values);

    chartBars.drawXaxisLabels(values);

    ChartCommon.drawChartAreaBorder(
      common.$chartAreaWithBorder,
      common.geom,
      '',
    );
  }
}

export default ChartBars;
