/* eslint-disable no-continue,no-restricted-properties,no-param-reassign */
import SvgUtils from '../svg/SvgUtils';
import ChartCommon from '../common/ChartCommon';
import BarsGeom from './BarsGeom';
import YAxisScaleData from '../common/YAxisScaleData';
import * as AggrFunctions from '../common/AggrFunctions';
import ChartValueHover from '../hover/ChartValueHover';

class ChartBarsCommon {
  static CHART_AXIS_LABEL_CSS_CLASS = 'chart-axis-label';

  /**
   * @param {CommonChartElements} commonChartElements
   * @param {BarsGeom} barsGeom
   */
  constructor(commonChartElements, barsGeom) {
    this.commonChartElements = commonChartElements;
    this.barsGeom = barsGeom;
  }

  killHover() {
    ChartValueHover.killHover(this.commonChartElements.$chartRootDiv);
  }

  /**
   * @param {GraphFormatter} graphFormatter
   * @param {AggrValue} value
   * @param {number} index
   */
  drawHoverInBarLeft(graphFormatter, value, index) {
    const chartAreaX = this.barsGeom.getBarLeft(index);
    this.drawHoverByPosition(graphFormatter, value, chartAreaX);
  }

  /**
   * @param {GraphFormatter} graphFormatter
   * @param {AggrValue} value
   * @param {number} chartAreaX
   */
  drawHoverByPosition(graphFormatter, value, chartAreaX) {
    const chartAreaLeftInChartRootDiv = this.commonChartElements.geom.getLeftYaxisAreaWidthOrZero()
      + this.commonChartElements.geom.chartBorderWidth;

    const left = chartAreaX + chartAreaLeftInChartRootDiv;
    const top = this.commonChartElements.geom.chartBorderWidth;
    const rightBoundary = chartAreaLeftInChartRootDiv
      + this.commonChartElements.geom.chartAreaWidth;

    ChartValueHover.drawHoverWithRightBoundary(
      value,
      left, top, rightBoundary,
      this.commonChartElements.$chartRootDiv,
      graphFormatter,
    );
  }

  valueToChartAreaY(value) {
    return Math.trunc(this.commonChartElements.geom.dataYToChartAreaY('left', value));
  }

  /**
   * @param {MouseEvent} event
   * @return {number}
   */
  cursorX(event) {
    return event.clientX - this.commonChartElements.geom.getChartAreaClientLeft();
  }

  drawXaxisLabels(values) {
    let minLabelLeft = 0;

    for (let i = 0; i < values.length; ++i) {
      const value = values[i];

      const chartXForText = this.commonChartElements.geom.chartBorderWidth
        + this.barsGeom.getBarMiddle(i);

      const labelText = value.label;

      const measureBox = SvgUtils.measureTextSize(labelText);

      const labelWidth = measureBox.width;
      const labelLeft = chartXForText - (labelWidth / 2);
      const labelRight = chartXForText + (labelWidth / 2);

      if (labelLeft < minLabelLeft || labelRight > this.commonChartElements.chartAreaWidth) {
        continue;
      }

      minLabelLeft = labelRight;

      const $text = SvgUtils.createTextElement();
      $text.setAttribute('class', ChartBarsCommon.CHART_AXIS_LABEL_CSS_CLASS);
      $text.setAttribute('x', `${chartXForText}`);
      $text.setAttribute('dy', '1em');
      $text.setAttribute('text-anchor', 'middle');
      SvgUtils.appendText($text, labelText);
      this.commonChartElements.$xaxisArea.appendChild($text);
    }
  }

  /**
   * @param {AggrValue[]} values
   * @param {ChartConf} conf
   * @return {YAxisScaleData}
   */
  static computeAggrYAxisMinMax(values, conf) {
    const yAxisMax = AggrFunctions.computePositiveMaxOrZero(values);
    const yAxisMin = AggrFunctions.computeNegativeMinOrZero(values);
    const yAxisMinAbs = AggrFunctions.computeMinAbs(values, (v) => v.value);

    let yAxisMaxPlusSome;
    let yAxisMinPlusSome;

    if (conf.yaxis.log) {
      yAxisMinPlusSome = Math.sign(yAxisMin) * Math.pow(Math.abs(yAxisMin), 1.05);
      yAxisMaxPlusSome = Math.sign(yAxisMax) * Math.pow(Math.abs(yAxisMax), 1.05);
    } else {
      let some = (yAxisMax - yAxisMin) * 0.025;
      if (some === 0) {
        some = 1;
      }

      // This 'some' gap must be added to both sides equally
      // if min and max are simultaneously zero or non-zero
      // And only to non-zero side if only one of them is zero.
      yAxisMaxPlusSome = yAxisMax;
      yAxisMinPlusSome = yAxisMin;
      if (yAxisMax === 0) {
        yAxisMinPlusSome -= some;
      } else {
        yAxisMaxPlusSome += some;
      }

      if (yAxisMin === 0) {
        yAxisMaxPlusSome += some;
      } else {
        yAxisMinPlusSome -= some;
      }
    }

    let yaxisScaleData = new YAxisScaleData(
      yAxisMinPlusSome,
      yAxisMaxPlusSome,
      yAxisMinAbs,
      yAxisMin,
    );

    yaxisScaleData = ChartCommon.adjustMinMaxWithConf(yaxisScaleData, conf.yaxis);
    return yaxisScaleData;
  }

  /**
   * @param {Element} $where
   * @param {AggrValue[]} values,
   * @param {ChartConf} conf,
   * @param {number} chartBorderWidth
   * @return {ChartBarsCommon}
   */
  static makeChartBarsCommon(
    $where,
    values,
    conf,
    chartBorderWidth,
  ) {
    $where.innerHTML = '';

    const yaxisScaleData = ChartBarsCommon.computeAggrYAxisMinMax(values, conf);

    const yaxisScaleDataMap = { left: yaxisScaleData };

    const commonChartElements = ChartCommon.makeCommonChartElements(
      $where, yaxisScaleDataMap, 0, conf, chartBorderWidth, false,
    );

    const barsGeom = new BarsGeom(commonChartElements.geom, values.length);

    return new ChartBarsCommon(commonChartElements, barsGeom);
  }
}

export default ChartBarsCommon;
