import SvgUtils from '../svg/SvgUtils';
import ChartCommon from '../common/ChartCommon';
import YaxisCommon from '../common/YaxisCommon';
import SvgDBuilder from '../common/SvgDBuilder';
import CursorHover from '../hover/CursorHover';

class ChartDistribution {
  static drawDistribution(chartBars, graphFormatter, values) {
    values.sort((value1, value2) => value1.value - value2.value);

    const { $chartArea } = chartBars.commonChartElements;
    const $path = SvgUtils.createPathElement();

    $path.setAttribute('stroke-width', '1px');
    $path.setAttribute('fill', 'none');
    $path.setAttribute('stroke', 'black');

    const d = ChartDistribution.makeDistributionD(chartBars, values);

    $path.setAttribute('d', d.format());

    $chartArea.appendChild($path);

    $chartArea.addEventListener('mousemove', (event) => {
      CursorHover.clearCursorAndPoints($chartArea);

      const x = chartBars.cursorX(event);
      const barIndexFromBarLeft = chartBars.barsGeom.getBarIndexFromBarLeftNoPadding(x);

      if (barIndexFromBarLeft >= 0 && barIndexFromBarLeft < values.length) {
        const aggrValue = values[barIndexFromBarLeft];
        chartBars.drawHoverByPosition(graphFormatter, aggrValue, x);
        CursorHover.drawCursor($chartArea, x);
        const y = chartBars.valueToChartAreaY(aggrValue.value);
        CursorHover.drawPoint($chartArea, x, y, aggrValue.color);
      }
    });

    $chartArea.addEventListener('mouseleave', () => {
      chartBars.killHover($chartArea);
      CursorHover.clearCursorAndPoints($chartArea);
    });
  }

  static makeDistributionD(chartBars, values) {
    if (ChartDistribution.isManyValues(chartBars, values)) {
      return ChartDistribution.makeDForManyValues(chartBars, values);
    }
    return ChartDistribution.makeDForNotManyValues(chartBars, values);
  }

  static makeDForManyValues(chartBars, values) {
    const dBuilder = new SvgDBuilder();
    for (let i = 0; i < values.length; i++) {
      const aggrValue = values[i];

      const xPos = chartBars.barsGeom.getBarLeftNoPadding(i);
      const yPos = chartBars.valueToChartAreaY(aggrValue.value);

      if (i === 0) {
        dBuilder.moveSharp(xPos, yPos, 1);
      } else {
        dBuilder.lineSharp(xPos, yPos, 1);
      }
    }
    return dBuilder;
  }

  static makeDForNotManyValues(chartBars, values) {
    const dBuilder = new SvgDBuilder();

    for (let i = 0; i < values.length; i++) {
      const aggrValue = values[i];

      const xPos = chartBars.barsGeom.getBarLeftNoPadding(i);
      const yPos = chartBars.valueToChartAreaY(aggrValue.value);

      if (i === 0) {
        dBuilder.moveSharp(0, yPos, 1);
      }

      dBuilder.lineSharp(xPos, yPos, 1);
      const nextXPos = chartBars.barsGeom.getBarLeftNoPadding(i + 1);
      dBuilder.lineSharp(nextXPos, yPos, 1);
    }

    return dBuilder;
  }

  static isManyValues(chartBars, values) {
    return values.length >= chartBars.commonChartElements.chartAreaWidth / 2;
  }

  /**
   *
   * @param {ChartBars} chartBars
   * @param {GraphFormatter} graphFormatter
   * @param {AggrValue[]} values,
   * @param {ChartConf} conf
   */
  static chart(chartBars, graphFormatter, values, conf) {
    const { commonChartElements } = chartBars;

    YaxisCommon.drawYAxisLabels(conf, commonChartElements);

    YaxisCommon.drawGridY(commonChartElements.geom, commonChartElements.$chartArea);

    ChartDistribution.drawDistribution(chartBars, graphFormatter, values);

    ChartCommon.drawChartAreaBorder(
      commonChartElements.$chartAreaWithBorder,
      commonChartElements.geom,
      '',
    );
  }
}

export default ChartDistribution;
