/* eslint-disable no-param-reassign,no-restricted-properties,no-restricted-syntax,guard-for-in */
/* eslint-disable import/no-cycle */
import SvgUtils from '../svg/SvgUtils';
import YaxisCommon from './YaxisCommon';
import CommonChartElements from './CommonChartElements';
import ChartGeom from './ChartGeom';
import parseKmg from '../../utils/Kmg';
import Multiplier from '../../utils/Multiplier';
import Mathx from '../../utils/Mathx';
import YAxisScaleData from './YAxisScaleData';

class ChartCommon {
  static CHART_ROOT_DIV_CLASS = 'chart-root-div';

  static DEFAULT_BORDER_COLOR = 'rgb(176,176,176)';

  /**
   * Create root div
   *
   * @param $where placeholder
   * @return {Element} created root div element
   */
  static createRootDiv($where) {
    $where.innerHTML = '<div class="chart-root-div" style="width: 100%; height: 100%; position: relative"></div>';
    return $where.getElementsByClassName(ChartCommon.CHART_ROOT_DIV_CLASS).item(0);
  }

  /**
   * Create root SVG element
   *
   * @param $chartRootDiv
   * @return {SVGElement} root SVG element
   */
  static createRootSvg($chartRootDiv) {
    const $svg = SvgUtils.createSvgElement();
    $svg.setAttribute('height', '100%');
    $svg.setAttribute('width', '100%');
    $chartRootDiv.appendChild($svg);
    return $svg;
  }

  /**
   * Make chart area with border
   *
   * @param {SVGElement} $rootSvg
   * @param {ChartGeom} geom
   * @return {*} chart area element with border
   */
  static makeChartAreaWithBorder($rootSvg, geom) {
    const $chartAreaWithBorder = SvgUtils.createSvgElement();
    $chartAreaWithBorder.setAttribute('x', geom.getLeftYaxisAreaWidthOrZero());
    $chartAreaWithBorder.setAttribute('y', '0');
    $chartAreaWithBorder.setAttribute('width', geom.chartAreaWithBorderWidth);
    $chartAreaWithBorder.setAttribute('height', geom.chartAreaWithBorderHeight);
    $rootSvg.appendChild($chartAreaWithBorder);
    return $chartAreaWithBorder;
  }

  /**
   * Make chart area
   *
   * @param {SVGElement} $chartAreaWithBorder
   * @param {ChartGeom} geom
   * @return {SVGElement} chart area
   */
  static makeChartArea(
    $chartAreaWithBorder,
    geom,
  ) {
    const $chartArea = SvgUtils.createSvgElement();
    $chartArea.setAttribute('class', 'chart-rect');
    $chartArea.setAttribute('x', `${geom.chartBorderWidth}`);
    $chartArea.setAttribute('y', `${geom.chartBorderWidth}`);
    $chartArea.setAttribute('width', `${geom.chartAreaWidth}`);
    $chartArea.setAttribute('height', `${geom.chartAreaHeight}`);
    $chartArea.setAttribute('pointer-events', 'all');

    $chartAreaWithBorder.appendChild($chartArea);

    {
      const rect = SvgUtils.createRectElement();

      // just to make the whole area clickable
      rect.setAttribute('width', '100%');
      rect.setAttribute('height', '100%');
      rect.setAttribute('fill', 'none');
      $chartArea.appendChild(rect);
    }

    return $chartArea;
  }

  /**
   * Make x-axis area
   *
   * @param {SVGElement} $rootSvg
   * @param {ChartGeom} geom
   * @return {SVGElement} new x-axis area svg element
   */
  static makeXaxisArea($rootSvg, geom) {
    const $xaxisArea = SvgUtils.createSvgElement();
    $xaxisArea.setAttribute('class', 'xaxis-area');
    $xaxisArea.setAttribute('x', geom.getLeftYaxisAreaWidthOrZero());
    $xaxisArea.setAttribute('y', geom.chartAreaWithBorderHeight);
    $xaxisArea.setAttribute('width', geom.chartAreaWithBorderWidth);
    $xaxisArea.setAttribute('height', geom.xaxisAreaHeight);
    $rootSvg.appendChild($xaxisArea);
    return $xaxisArea;
  }

  /**
   * Format value with suffix
   * @param {number} value
   * @return {string}
   */
  static formatValueWithSuffix(value) {
    if (isNaN(value)) {
      return 'No data';
    }
    const sign = value < 0 ? '-' : '';
    value = Math.abs(value);
    const m = Multiplier.getMultiplierForNumber(value);
    return sign + ChartCommon.formatDoublePossibleInteger(value / m.c) + m.suffix;
  }

  /**
   * Format double possible integer
   * @param {number} value
   * @return {string}
   */
  static formatDoublePossibleInteger(value) {
    return value.toFixed(3).replace('\\.?0*$', '');
  }

  static roundUpTo2510(value) {
    const r10 = Mathx.roundUp10(value);
    const r5 = Math.pow(10, Math.ceil(Math.log10(value / 2))) * 2;
    const r2 = Math.pow(10, Math.ceil(Math.log10(value / 5))) * 5;
    return Math.min(r10, Math.min(r5, r2));
  }

  /**
   * Make common chart elements
   *
   * @param {Element} $chartRootDiv
   * @param {object} yaxisScaleDataMap
   * @param {number} xaxisRange
   * @param {object} conf
   * @param {number} chartBorderWidth
   * @param {boolean} hasMinYaxisAreaWidth
   * @return {CommonChartElements}
   */
  static makeCommonChartElements(
    $chartRootDiv,
    yaxisScaleDataMap,
    xaxisRange,
    conf,
    chartBorderWidth,
    hasMinYaxisAreaWidth,
  ) {
    const $rootSvg = ChartCommon.createRootSvg($chartRootDiv);

    const minYaxisAreaWidth = hasMinYaxisAreaWidth
      ? YaxisCommon.MIN_YAXIS_AREA_WIDTH_BY_DEFAULT
      : 0;

    let geom = new ChartGeom(
      $chartRootDiv.scrollWidth,
      $chartRootDiv.scrollHeight,
      // because $chartArea.offset() returns wrong on Chrome
      Math.trunc($chartRootDiv.getBoundingClientRect().left),
      0,
      chartBorderWidth,
      0,
      0,
      {},
      true,
      minYaxisAreaWidth,
    );

    const xaxisAreaHeight = Math.trunc(SvgUtils.measureTextSize('1A').height + 2);
    geom = geom.addXaxis(xaxisAreaHeight, conf.xaxis.min, xaxisRange);

    for (const yaxisKey in yaxisScaleDataMap) {
      geom = geom.addYaxis(yaxisKey, yaxisScaleDataMap[yaxisKey], conf.yaxis.log);
    }

    if (!YaxisCommon.showYAxis(conf.yaxis)) {
      geom = geom.hideYaxis();
    }

    const $chartAreaWithBorder = ChartCommon.makeChartAreaWithBorder($rootSvg, geom);

    const $chartArea = ChartCommon.makeChartArea($chartAreaWithBorder, geom);
    const $xaxisArea = ChartCommon.makeXaxisArea($rootSvg, geom);
    const $yaxisAreas = YaxisCommon.makeYaxisAreas($rootSvg, geom);

    return new CommonChartElements(
      $chartRootDiv,
      $chartAreaWithBorder,
      $chartArea,
      $xaxisArea,
      $yaxisAreas,
      geom,
    );
  }

  /**
   * Draw chart area border
   * @param {SVGElement} $chartAreaWithBorder
   * @param {ChartGeom} geom
   * @param {string} borderColor
   */
  static drawChartAreaBorder($chartAreaWithBorder, geom, borderColor) {
    const $rect = SvgUtils.createRectElement();
    $rect.setAttribute('class', 'chart-rect');
    $rect.setAttribute('x', geom.chartBorderWidth / 2);
    $rect.setAttribute('y', geom.chartBorderWidth / 2);
    $rect.setAttribute('width', geom.chartAreaWithBorderWidth - geom.chartBorderWidth);
    $rect.setAttribute('height', geom.chartAreaWithBorderHeight - geom.chartBorderWidth);
    $rect.setAttribute('fill', 'none');
    $rect.setAttribute('stroke', borderColor || ChartCommon.DEFAULT_BORDER_COLOR);
    $rect.setAttribute('stroke-width', geom.chartBorderWidth);
    $chartAreaWithBorder.appendChild($rect);
  }

  /**
   * Adjust min max with config
   *
   * @param {YAxisScaleData} r
   * @param {ChartConfYAxis} yAxis
   * @return {YAxisScaleData}
   */
  static adjustMinMaxWithConf(r, yAxis) {
    let min = r.getMin();
    let max = r.getMax();
    const minAbs = r.getMinAbs();

    if (yAxis.min && yAxis.min !== 'datamin') {
      min = parseKmg(yAxis.min);
    }

    if (yAxis.max != null && yAxis.max) {
      max = parseKmg(yAxis.max);
    }

    if (min > max) {
      min = max;
    }

    return new YAxisScaleData(min, max, minAbs, min);
  }

  /**
   * Filter show
   * @param {AggrValue[]} values
   * @return {AggrValue[]}
   */
  static filterShow(values) {
    return values.filter((v) => v.show);
  }

  static filterNotNan(values) {
    return values.filter((v) => !isNaN(v.value));
  }

  static filterNotNanAndNotZero(values) {
    return values.filter((v) => !isNaN(v.value) && v.value !== 0);
  }
}

export default ChartCommon;
