/* eslint-disable guard-for-in,no-restricted-syntax,import/no-cycle */
import Axis from './Axis';

class ChartGeom {
  /**
   * @param {number} svgWidth
   * @param {number} svgHeight
   * @param {number} svgOffsetLeft
   * @param {number} xaxisAreaHeight
   * @param {number} chartBorderWidth
   * @param {number} xaxisMin
   * @param {number} xaxisRange
   * @param {object} yaxisByPosition
   * @param {boolean} showYaxis
   * @param {number} minYaxisAreaWidth
   */
  constructor(
    svgWidth,
    svgHeight,
    svgOffsetLeft,
    xaxisAreaHeight,
    chartBorderWidth,
    xaxisMin,
    xaxisRange,
    yaxisByPosition,
    showYaxis,
    minYaxisAreaWidth,
  ) {
    this.chartBorderWidth = chartBorderWidth;
    this.xaxisAreaHeight = xaxisAreaHeight;
    this.chartAreaWithBorderHeight = svgHeight - xaxisAreaHeight;
    this.chartAreaHeight = this.chartAreaWithBorderHeight - (2 * chartBorderWidth);

    const recalculatedYaxisByPosition = ChartGeom.makeYaxisMapWithRecalculatedWidthAndHeight(
      yaxisByPosition,
      this.chartAreaHeight,
      minYaxisAreaWidth,
    );

    this.showYaxis = showYaxis;
    this.minYaxisAreaWidth = minYaxisAreaWidth;

    this.yaxisAreasSummaryWidth = showYaxis ? ChartGeom.sumYaxisAreasWidth(yaxisByPosition) : 0;
    this.chartAreaWithBorderWidth = svgWidth - this.yaxisAreasSummaryWidth;

    this.chartAreaWidth = this.chartAreaWithBorderWidth - (2 * chartBorderWidth);

    this.leftYaxisAreaWidth = showYaxis
      ? ChartGeom.leftYaxisAreaWidthOr(yaxisByPosition, minYaxisAreaWidth)
      : 0;

    this.chartAreaOffsetLeft = this.leftYaxisAreaWidth + chartBorderWidth;

    this.yaxisByPosition = ChartGeom.makeYaxisMapWithRecalculatedOffset(
      recalculatedYaxisByPosition,
      this.chartAreaWidth,
      chartBorderWidth,
      minYaxisAreaWidth,
    );

    this.xaxisMin = xaxisMin;
    this.xaxisRange = xaxisRange;

    this.svgWidth = svgWidth;
    this.svgHeight = svgHeight;
    this.svgOffsetLeft = svgOffsetLeft;
  }

  getYaxisAreasSummaryWidth() {
    return this.yaxisAreasSummaryWidth;
  }

  getChartAreaClientLeft() {
    return this.svgOffsetLeft + this.chartAreaOffsetLeft;
  }

  getLeftYaxisAreaWidthOrZero() {
    return this.leftYaxisAreaWidth;
  }

  /**
   * @param {object} yaxisMap
   * @param {number} minYaxisAreaWidth
   * @return {number}
   */
  static leftYaxisAreaWidthOr(yaxisMap, minYaxisAreaWidth) {
    const yaxis = yaxisMap.left;
    return !yaxis ? minYaxisAreaWidth : yaxis.areaWidth;
  }

  static sumYaxisAreasWidth(yaxisMap) {
    let yaxisAreaWidth = 0;
    for (const yaxisName in yaxisMap) {
      const axis = yaxisMap[yaxisName];
      yaxisAreaWidth += !axis ? 0 : axis.areaWidth;
    }
    return yaxisAreaWidth;
  }

  static makeYaxisMapWithRecalculatedWidthAndHeight(yaxisMap, chartAreaHeight, minYaxisAreaWidth) {
    const newYaxisMap = {};

    for (const axisName in yaxisMap) {
      const axis = yaxisMap[axisName];
      newYaxisMap[axisName] = new Axis(
        axis.min, axis.max, axis.isLog(),
        chartAreaHeight,
        axis.minAbs, axis.minValue,
        0,
        minYaxisAreaWidth,
      );
    }

    return newYaxisMap;
  }

  static makeYaxisMapWithRecalculatedOffset(
    yaxisMap,
    chartAreaWidth,
    borderWidth,
    minYaxisAreaWidth,
  ) {
    const leftYaxisAreaWidth = ChartGeom.leftYaxisAreaWidthOr(yaxisMap, minYaxisAreaWidth);

    const newYaxisMap = {};
    for (const axisName in yaxisMap) {
      const axis = yaxisMap[axisName];

      const offsetLeft = axisName === 'left'
        ? 0
        : leftYaxisAreaWidth + borderWidth + chartAreaWidth + borderWidth;

      newYaxisMap[axisName] = axis.copyWithNewOffsetLeft(offsetLeft);
    }

    return newYaxisMap;
  }

  /**
   * Convert data x to chart area x
   * @param {number} dataX
   * @return {number} x position in area coordinates
   */
  dataXToChartAreaX(dataX) {
    // prevent buggy Safari optimization (?)
    // that causes problems like this: http://jing.yandex-team.ru/files/nga/2015-06-22_2317.png
    if (this.chartAreaWidth === 0) {
      throw new Error('buggy bug');
    }

    const result = ((dataX - this.xaxisMin) / this.xaxisRange) * this.chartAreaWidth;

    return Math.trunc(result);
  }

  /**
   * Convert chart area x to data x
   * @param {number} areaX
   * @return {number} x position in data coordinates
   */
  chartAreaXToDataX(areaX) {
    return ((areaX / this.chartAreaWidth) * this.xaxisRange) + this.xaxisMin;
  }

  /**
   * @param {String} position
   * @param {number} dataY
   * @return {number} y position in chart area coordinates
   */
  dataYToChartAreaY(position, dataY) {
    const yaxis = this.yaxisByPosition[position];
    return yaxis ? yaxis.dataYToChartAreaY(dataY) : 0;
  }

  getYaxis(position) {
    return this.yaxisByPosition[position];
  }

  addXaxis(xaxisAreaHeight, xaxisMin, xaxisRange) {
    return new ChartGeom(
      this.svgWidth,
      this.svgHeight,
      this.svgOffsetLeft,
      xaxisAreaHeight,
      this.chartBorderWidth, xaxisMin,
      xaxisRange,
      this.yaxisByPosition,
      this.showYaxis,
      this.minYaxisAreaWidth,
    );
  }

  addYaxis(position, yAxisScaleData, log) {
    const yaxisMap = { ...this.yaxisByPosition };
    yaxisMap[position] = ChartGeom.convertYAxisScaleDataToAxis(
      yAxisScaleData,
      log,
      this.chartAreaHeight,
      this.minYaxisAreaWidth,
    );

    return new ChartGeom(
      this.svgWidth,
      this.svgHeight,
      this.svgOffsetLeft,
      this.xaxisAreaHeight,
      this.chartBorderWidth, this.xaxisMin,
      this.xaxisRange,
      yaxisMap,
      this.showYaxis,
      this.minYaxisAreaWidth,
    );
  }

  hideYaxis() {
    return new ChartGeom(
      this.svgWidth,
      this.svgHeight,
      this.svgOffsetLeft,
      this.xaxisAreaHeight,
      this.chartBorderWidth, this.xaxisMin,
      this.xaxisRange,
      this.yaxisByPosition,
      false,
      this.minYaxisAreaWidth,
    );
  }

  getYaxisMap() {
    return this.yaxisByPosition;
  }

  /**
   * @param {YAxisScaleData} yAxisScaleData
   * @param {boolean} log
   * @param {number} chartAreaHeight
   * @param {number} minYaxisAreaWidth
   * @return {Axis} new axis
   */
  static convertYAxisScaleDataToAxis(
    yAxisScaleData,
    log,
    chartAreaHeight,
    minYaxisAreaWidth,
  ) {
    return new Axis(
      yAxisScaleData.getMin(),
      yAxisScaleData.getMax(),
      log,
      chartAreaHeight,
      yAxisScaleData.getMinAbs(),
      yAxisScaleData.getMinValue(),
      0,
      minYaxisAreaWidth,
    );
  }
}

export default ChartGeom;
