/* eslint-disable no-param-reassign */
class LogYAxisTransformer {
  static ZERO_AREA_MARGIN = 4;

  /**
   * @param {number} chartAreaHeight
   * @param {number} yaxisMin
   * @param {number} yaxisMax
   * @param {number} yaxisMinAbs
   */
  constructor(
    chartAreaHeight,
    yaxisMin,
    yaxisMax,
    yaxisMinAbs,
  ) {
    this.chartAreaHeight = chartAreaHeight;

    if (yaxisMinAbs === 0) {
      // == only zeroes
      this.zeroPosition = chartAreaHeight * 0.5;
    } else {
      let totalTicks;
      let numTickIntervalsPositive;
      let zeroAreaHeight;
      // Number of ticks from minimal tick to axis-min and axis-max
      let ticksToMin;
      let ticksToMax;

      // if +-smallest value is not in [min..max]
      // (which only can happen if min and max have similar sign?)
      // then [min..max] was set explicitly and we have to consider it
      if (yaxisMin * yaxisMax > 0) {
        yaxisMinAbs = Math.max(yaxisMinAbs, Math.min(Math.abs(yaxisMin), Math.abs(yaxisMax)));
      }

      if (yaxisMax === 0) {
        ticksToMax = 0;
      } else {
        ticksToMax = Math.log10(Math.abs(yaxisMax)) - Math.log10(yaxisMinAbs);
      }
      if (yaxisMin === 0) {
        ticksToMin = 0;
      } else {
        ticksToMin = Math.log10(Math.abs(yaxisMin)) - Math.log10(yaxisMinAbs);
      }

      if (yaxisMax > 0) {
        if (yaxisMin < 0) {
          totalTicks = ticksToMax + ticksToMin;
          numTickIntervalsPositive = ticksToMax;
          zeroAreaHeight = 1 + (2 * LogYAxisTransformer.ZERO_AREA_MARGIN);
        } else {
          totalTicks = ticksToMax - ticksToMin;
          numTickIntervalsPositive = totalTicks;
          zeroAreaHeight = 1 + LogYAxisTransformer.ZERO_AREA_MARGIN;
        }
      } else {
        totalTicks = ticksToMin - ticksToMax;
        numTickIntervalsPositive = 0;
        zeroAreaHeight = 1 + LogYAxisTransformer.ZERO_AREA_MARGIN;
      }

      let tickInterval = (chartAreaHeight - zeroAreaHeight) / totalTicks;

      if (isNaN(tickInterval) || !Number.isFinite(tickInterval)) {
        tickInterval = chartAreaHeight - zeroAreaHeight;
        numTickIntervalsPositive = 1;
      }

      this.zeroPosition = (tickInterval * numTickIntervalsPositive)
        + ((numTickIntervalsPositive > 0) ? LogYAxisTransformer.ZERO_AREA_MARGIN : 0);
    }
    this.yaxisMin = yaxisMin;
    this.yaxisMax = yaxisMax;
    this.yaxisMinAbs = yaxisMinAbs;
  }

  /**
   * Convert data y to chart area y
   * @param {number} dataY data y
   * @return {number} y in chart area coordinates
   */
  yToChartY(dataY) {
    if ((this.yaxisMinAbs === 0) || (Math.abs(dataY) < this.yaxisMinAbs)) {
      return this.zeroPosition;
    }

    if (dataY > 0) {
      let v01 = (Math.log10(dataY) - Math.log10(this.yaxisMinAbs))
        / (Math.log10(this.yaxisMax) - Math.log10(this.yaxisMinAbs));

      if (v01 < 0 || isNaN(v01)) {
        v01 = 0;
      }

      return (1 - v01) * (this.zeroPosition - LogYAxisTransformer.ZERO_AREA_MARGIN);
    }

    let v01 = (Math.log10(-dataY) - Math.log10(this.yaxisMinAbs))
      / (Math.log10(-this.yaxisMin) - Math.log10(this.yaxisMinAbs));

    if (v01 < 0 || isNaN(v01)) {
      v01 = 0;
    }

    const chartHeightWithoutZero = this.chartAreaHeight
      - this.zeroPosition - LogYAxisTransformer.ZERO_AREA_MARGIN - 1;

    return this.zeroPosition
      + LogYAxisTransformer.ZERO_AREA_MARGIN
      + (chartHeightWithoutZero * v01);
  }
}

export default LogYAxisTransformer;
