/* eslint-disable guard-for-in,no-restricted-syntax,no-continue,no-param-reassign,import/no-cycle */
import isEmpty from 'lodash/isEmpty';
import SvgUtils from '../svg/SvgUtils';
import Multiplier from '../../utils/Multiplier';
import Mathx from '../../utils/Mathx';
import ChartCommon from './ChartCommon';
import Axis from './Axis';
import SvgDBuilder from './SvgDBuilder';

class YaxisCommon {
  static MIN_YAXIS_TICK_HEIGHT = 30;

  static MIN_YAXIS_AREA_WIDTH_BY_DEFAULT = 44;

  static MAX_TICKS_COUNT = 1000;

  /**
   * @param {{ y: number, yFormatted: string }[]} yaxisTicks
   * @param {number} minYaxisAreaWidth
   * @return {number}
   */
  static computeYaxisAreaWidth(yaxisTicks, minYaxisAreaWidth) {
    let yaxisAreaWidth = minYaxisAreaWidth;
    for (let i = 0; i < yaxisTicks.length; ++i) {
      const tick = yaxisTicks[i];
      const size = SvgUtils.measureTextSize(tick.yFormatted);
      const tickWidth = size.width;
      yaxisAreaWidth = Math.max(yaxisAreaWidth, Math.trunc(tickWidth + 2));
    }
    return yaxisAreaWidth;
  }

  /**
   * @param {SVGElement} $rootSvg
   * @param {ChartGeom} geom
   * @return {object}
   */
  static makeYaxisAreas($rootSvg, geom) {
    const result = {};

    const yaxisMap = geom.getYaxisMap();
    for (const position in yaxisMap) {
      const axis = yaxisMap[position];
      const $axis = SvgUtils.createSvgElement();
      $axis.setAttribute('class', 'yaxis-area');
      $axis.setAttribute('width', `${axis.areaWidth}`);
      $axis.setAttribute('x', `${axis.offsetLeft}`);
      $axis.setAttribute('height', '100%');
      $rootSvg.appendChild($axis);
      result[position] = $axis;
    }

    return result;
  }

  /**
   * @param {number} value
   * @param {number} tickSize
   * @return {string}
   */
  static formatYTickValue2(value, tickSize) {
    if (tickSize >= 1) {
      const m = Multiplier.getMultiplierForNumber(tickSize);
      const prefix = value / m.c;
      return prefix.toFixed(m.digitsAfterPointOverride) + m.suffix;
    }
    const nDigits = -Mathx.floorLog10(tickSize);

    if (nDigits > 9) {
      return value.toFixed(9);
    }

    return value.toFixed(nDigits);
  }

  /**
   * @param {number} minHeight
   * @param {number} valueRange
   * @param {number} areaHeight
   * @return {number}
   */
  static computeYaxisTickHeight(minHeight, valueRange, areaHeight) {
    const valuePerPx = valueRange / areaHeight;
    const valuePerMinHeight = minHeight * valuePerPx;
    return ChartCommon.roundUpTo2510(valuePerMinHeight);
  }

  /**
   * @param {number} minHeight
   * @param {number} yaxisMin
   * @param {number} yaxisMax
   * @param {number} chartAreaHeight
   * @return {{ y: number, yFormatted: string }[]}
   */
  static makeYTicksLinear(
    minHeight,
    yaxisMin,
    yaxisMax,
    chartAreaHeight,
  ) {
    const r = [];
    const tickHeightValue = YaxisCommon.computeYaxisTickHeight(
      minHeight,
      yaxisMax - yaxisMin,
      chartAreaHeight,
    );
    let y = Mathx.roundUp(yaxisMin, tickHeightValue);
    let ticksCount = 0;
    while (y < yaxisMax) {
      const yFormatted = YaxisCommon.formatYTickValue2(y, tickHeightValue);
      r.push({ y, yFormatted });
      y += tickHeightValue;
      ticksCount += 1;
      if (ticksCount > this.MAX_TICKS_COUNT) {
        break;
      }
    }
    return r;
  }

  /**
   * @param {YAxisScaleData} yaxisScaleData
   * @return {{ y: number, yFormatted: string }[]}
   */
  static makeYTicksLog(yaxisScaleData) {
    const yaxisMin = yaxisScaleData.getMin();
    const yaxisMax = yaxisScaleData.getMax();
    const yaxisSmallest = yaxisScaleData.getMinAbs();

    const r = [];

    const doubles = YaxisCommon.makeYTickValuesLog(yaxisMin, yaxisMax, yaxisSmallest);

    for (const y of doubles) {
      let tick;

      if (y === 0) {
        // No label for zero on log scale for readability
        tick = { y: 0, yFormatted: '' };
      } else {
        const yFormatted = YaxisCommon.formatYTickValue2(y, Math.abs(y));
        tick = { y, yFormatted };
      }

      r.push(tick);
    }

    return r;
  }

  /**
   * @param {ChartConf} conf
   * @param {CommonChartElements} commonChartElements
   */
  static drawYAxisLabels(conf, commonChartElements) {
    if (YaxisCommon.showYAxis(conf.yaxis)) {
      YaxisCommon.drawYAxisLabels2(commonChartElements.$yaxisAreas, commonChartElements.geom);
    }
  }

  /**
   * @param {ChartConfYAxis} yaxis
   * @return {boolean}
   */
  static showYAxis(yaxis) {
    return yaxis.yaxis !== 'none';
  }

  /**
   * @param {array} lines
   * @return {object}
   */
  static groupLinesByYaxisPosition(lines) {
    const linesGroupedByYaxisName = {};

    lines.reduce((result, line) => {
      const position = line.yaxisConf.positionValue;

      let linesWithEqualYaxis = result[position];
      if (!linesWithEqualYaxis) {
        linesWithEqualYaxis = [];
        result[position] = linesWithEqualYaxis;
      }

      linesWithEqualYaxis.push(line);
      return result;
    }, linesGroupedByYaxisName);

    return linesGroupedByYaxisName;
  }

  /**
   * @param {number} yaxisMin
   * @param {number} yaxisMax
   * @param {number} yaxisMinAbs
   * @return {number[]}
   */
  static makeYTickValuesLog(
    yaxisMin,
    yaxisMax,
    yaxisMinAbs,
  ) {
    const yaxisMaxAbs = Math.max(Math.abs(yaxisMin), Math.abs(yaxisMax));
    const ticks = new Set();

    if (yaxisMinAbs !== 0) {
      let y = Mathx.roundDown10(yaxisMinAbs);
      if (y === 0) {
        // In case of too small values
        y = Mathx.roundUp10(yaxisMaxAbs);
      }

      let i = 0;

      while (i < YaxisCommon.MAX_TICKS_COUNT && y <= yaxisMaxAbs) {
        if ((y >= yaxisMin) && (y <= yaxisMax)) {
          ticks.add(y);
        }
        if ((-y >= yaxisMin) && (-y <= yaxisMax)) {
          ticks.add(-y);
        }

        y *= 10;

        i += 1;
      }

      if (i >= YaxisCommon.MAX_TICKS_COUNT) {
        return [];
      }
    }

    ticks.add(0.0);

    return Array.from(ticks).sort((a, b) => a - b);
  }

  /**
   * @param {object} $yaxisAreas
   * @param {ChartGeom} geom
   */
  static drawYAxisLabels2($yaxisAreas, geom) {
    for (const position in geom.getYaxisMap()) {
      const yaxis = geom.getYaxis(position);
      if (!yaxis) {
        continue;
      }

      let textAnchor;
      let chartXAsDouble;

      switch (position) {
        case 'right':
          chartXAsDouble = 1;
          textAnchor = 'start';
          break;
        default:
          chartXAsDouble = yaxis.areaWidth - 1;
          textAnchor = 'end';
      }
      const chartX = `${chartXAsDouble}`;

      const $yaxisArea = $yaxisAreas[position];
      const yaxisTicks = yaxis.ticks;
      for (let i = 0; i < yaxisTicks.length; ++i) {
        const tick = yaxisTicks[i];
        const chartY = yaxis.dataYToChartAreaY(tick.y) + geom.chartBorderWidth;

        const { yFormatted } = tick;

        const $text = SvgUtils.createTextElement();
        $text.setAttribute('class', 'chart-axis-label');
        $text.setAttribute('dy', '.4em'); // poor man vertical-align: center
        $text.setAttribute('x', chartX);
        $text.setAttribute('y', `${chartY}`);
        $text.setAttribute('text-anchor', textAnchor);

        $text.appendChild(document.createTextNode(yFormatted));

        $yaxisArea.appendChild($text);
      }
    }
  }

  /**
   * @param {YAxisScaleData} yaxisScaleData
   * @param {number} minHeight
   * @param {number} chartAreaHeight
   * @param {boolean} log
   * @return {{ y: number, yFormatted: string }[]}
   */
  static makeYAxisTicks(
    yaxisScaleData,
    minHeight,
    chartAreaHeight,
    log,
  ) {
    if (log) {
      return YaxisCommon.makeYTicksLog(yaxisScaleData);
    }

    return YaxisCommon.makeYTicksLinear(
      minHeight,
      yaxisScaleData.getMin(),
      yaxisScaleData.getMax(),
      chartAreaHeight,
    );
  }

  /**
   * @param {ChartGeom} geom
   * @param {SVGElement} $chartArea
   */
  static drawGridY(geom, $chartArea) {
    const yaxisMap = geom.getYaxisMap();
    const useFakeYaxis = isEmpty(yaxisMap);

    // order is important, left axis is first
    const positions = ['left', 'right'];

    for (const position of positions) {
      let yaxis = yaxisMap[position];
      if (!yaxis) {
        if (!useFakeYaxis) {
          continue;
        }

        yaxis = new Axis(
          -1,
          1,
          false,
          geom.chartAreaHeight,
          0,
          0,
          0,
          geom.minYaxisAreaWidth,
        );
      }

      const yaxisTicks = yaxis.ticks;

      for (let i = 0; i < yaxisTicks.length; ++i) {
        const tick = yaxisTicks[i];
        const chartAreaY = Math.trunc(yaxis.dataYToChartAreaY(tick.y));

        const d = new SvgDBuilder();
        d.moveSharp(0, chartAreaY, 1);
        d.lineH(10000);

        const $path = SvgUtils.createPathElementWithD('chart-grid', d);
        $chartArea.appendChild($path);
      }

      // we draw only one grid
      return;
    }
  }
}

export default YaxisCommon;
