/* eslint-disable no-param-reassign,no-continue,no-plusplus */
import Mathx from '../../utils/Mathx';
import { computeMinAbs } from '../common/AggrFunctions';
import { getGradientColorByZone } from '../../utils/color/colors';
import GradientZone from '../../utils/color/GradientZone';
import UserLinksBasic from '../../utils/UserLinksBasic';
import ChartValueHover from '../hover/ChartValueHover';

const colorNames = ['green', 'yellow', 'red', 'violet'];
const gray = '#090909';

function makeColorsArray(colors) {
  return [
    colors.green,
    colors.yellow,
    colors.red,
    colors.violet,
  ];
}

class ChartHeatmap {
  static computeSquareSize(count, areaWidth, areaHeight) {
    const area = areaWidth * areaHeight;
    const elementArea = area / count;
    let elementSizeUpperBound = Math.floor(Math.sqrt(elementArea)) + 1;
    let elementSizeLowerBound = 1;
    while (elementSizeLowerBound + 1 < elementSizeUpperBound) {
      const midElementSize = Math.trunc((elementSizeLowerBound + elementSizeUpperBound) / 2);
      const widthInElements = Math.trunc(areaWidth / midElementSize);
      const heightInElements = Math.trunc(areaHeight / midElementSize);
      if (widthInElements * heightInElements >= count) {
        elementSizeLowerBound = midElementSize;
      } else {
        elementSizeUpperBound = midElementSize;
      }
    }
    return elementSizeLowerBound;
  }

  static computeHeatmapColor(value, colors, isLog, minAbs) {
    if (isNaN(value)) {
      return gray;
    }

    if (isLog) {
      value = Mathx.pseudoLog(value, minAbs);
    }

    const order = colors.green < colors.violet;
    if ((value < colors.green) === order) {
      return getGradientColorByZone(0, GradientZone.GREEN_YELLOW_VALUE);
    }

    if ((value < colors.yellow) === order) {
      const koef = (value - colors.green) / (colors.yellow - colors.green);
      return getGradientColorByZone(koef, GradientZone.GREEN_YELLOW_VALUE);
    }

    if ((value < colors.red) === order) {
      const koef = (value - colors.yellow) / (colors.red - colors.yellow);
      return getGradientColorByZone(koef, GradientZone.YELLOW_RED_VALUE);
    }

    if ((value < colors.violet) === order) {
      const koef = (value - colors.red) / (colors.violet - colors.red);
      return getGradientColorByZone(koef, GradientZone.RED_VIOLET_VALUE);
    }

    return getGradientColorByZone(1, GradientZone.RED_VIOLET_VALUE);
  }

  static buildHeatmapColors(basicColors, isLog, data) {
    let min = NaN;
    let max = NaN;
    const minAbs = computeMinAbs(data, (v) => v.value);

    for (let i = 0; i < data.length; ++i) {
      const v = data[i].value;
      if (isNaN(v)) {
        continue;
      }

      if (isNaN(min) || v < min) {
        min = v;
      }
      if (isNaN(max) || v > max) {
        max = v;
      }
    }

    return ChartHeatmap.fillColorGaps(basicColors, min, max, minAbs, isLog);
  }

  /**
   * @param {GraphFormatter} graphFormatter
   * @param {Element} $div
   * @param {AggrValue[]} graphData
   * @param {ChartHeatmapConf} graphConf
   * @return {ChartHeatmap}
   */
  static chart(graphFormatter, $div, graphData, graphConf) {
    const { isLog } = graphConf;

    // eslint-disable-next-line max-len
    const heatmapColorValues = ChartHeatmap.buildHeatmapColors(graphConf.heatmapColorValues, isLog, graphData);

    let minAbs = 0;
    if (isLog) {
      minAbs = computeMinAbs(graphData, (v) => v.value);
    }

    const elements = [];
    const order = heatmapColorValues.green < heatmapColorValues.violet;

    for (let i = 0; i < graphData.length; ++i) {
      const graphLine = graphData[i];
      if (graphLine.show) {
        // eslint-disable-next-line max-len
        const color = ChartHeatmap.computeHeatmapColor(graphLine.value, heatmapColorValues, isLog, minAbs);

        const value = {
          show: true,
          label: graphLine.label || '',
          link: graphLine.link || '',
          value: graphLine.value,
          color,
        };

        elements.push(value);
      }
    }

    // eslint-disable-next-line max-len
    const size = ChartHeatmap.computeSquareSize(elements.length, $div.scrollWidth, $div.scrollHeight);

    elements.sort((l, r) => {
      let res;
      if (isNaN(l.value) && isNaN(r.value)) {
        res = 0;
      } else if (isNaN(l.value)) {
        res = -1;
      } else if (isNaN(r.value)) {
        res = 1;
      } else {
        res = l.value - r.value;
      }
      return order ? res : -res;
    });

    for (let i = 0; i < elements.length; i++) {
      elements[i].cookie = i;
    }

    let divSize = size - 1;
    if (divSize <= 0) {
      divSize = 1;
    }
    const divMargin = size - divSize;

    const $heatmapDiv = document.createElement('div');
    $heatmapDiv.setAttribute('class', 'heatmap');
    $heatmapDiv.setAttribute('style', 'line-height: 0; font-size: 0');

    for (let i = 0; i < elements.length; i++) {
      const aggrValue = elements[i];

      let $heatmapContainer = null;
      if (!aggrValue.link) {
        $heatmapContainer = $heatmapDiv;
      } else {
        $heatmapContainer = document.createElement('a');
        $heatmapContainer.style.display = 'inline-block';
        $heatmapContainer.setAttribute('href', UserLinksBasic.fixOldAdminUrl(aggrValue.link));
        $heatmapDiv.appendChild($heatmapContainer);
      }

      const cookie = aggrValue.cookie ? `${aggrValue.cookie}` : '';

      const $filledArea = document.createElement('div');
      $filledArea.setAttribute('class', 'container-heatmap');
      $filledArea.setAttribute('data-solomon-value-index', cookie);
      $filledArea.style.display = 'inline-block';
      $filledArea.style.margin = `0 ${divMargin}px ${divMargin}px 0`;
      $filledArea.style.minWidth = `${divSize}px`;
      $filledArea.style.minHeight = `${divSize}px`;
      $filledArea.style.backgroundColor = aggrValue.color;
      $filledArea.style.overflow = 'hidden';
      $heatmapContainer.appendChild($filledArea);

      $filledArea.addEventListener('mouseenter', () => {
        const rightBoundary = $div.clientWidth + $div.offsetLeft;
        const hoverLeft = $filledArea.offsetLeft;
        const hoverTop = $filledArea.offsetTop;

        ChartValueHover.drawHoverWithRightBoundary(
          aggrValue,
          hoverLeft, hoverTop, rightBoundary,
          $div, graphFormatter,
        );
      });

      $filledArea.addEventListener('mouseleave', () => {
        ChartValueHover.killHover($div);
      });
    }

    $div.appendChild($heatmapDiv);

    return new ChartHeatmap();
  }

  static getColorOrder(colors) {
    let firstValue = NaN;
    let lastValue = NaN;

    const colorsArray = makeColorsArray(colors);
    for (let i = 0; i < colorsArray.length; ++i) {
      const v = colorsArray[i];

      if (isNaN(v)) {
        continue;
      }
      if (isNaN(firstValue)) {
        firstValue = v;
      }
      lastValue = v;
    }

    if (!isNaN(firstValue) && !isNaN(lastValue)) {
      return firstValue - lastValue;
    }

    return 0;
  }

  static fillColorGaps(basicColors, min, max, minAbs, isLog) {
    if (isNaN(min) || isNaN(max)) {
      return basicColors;
    }

    if (isLog) {
      // Fixing color zones set in url for log scale
      if (!isNaN(basicColors.green)) {
        basicColors.green = Mathx.pseudoLog(basicColors.green, minAbs);
      }
      if (!isNaN(basicColors.yellow)) {
        basicColors.yellow = Mathx.pseudoLog(basicColors.yellow, minAbs);
      }
      if (!isNaN(basicColors.red)) {
        basicColors.red = Mathx.pseudoLog(basicColors.red, minAbs);
      }
      if (!isNaN(basicColors.violet)) {
        basicColors.violet = Mathx.pseudoLog(basicColors.violet, minAbs);
      }

      if (minAbs === 0) {
        min = 0;
        max = 0;
      } else {
        min = Mathx.pseudoLog(min, minAbs);
        max = Mathx.pseudoLog(max, minAbs);
      }
    }
    const order = ChartHeatmap.getColorOrder(basicColors);
    if (order > 0) {
      if (isNaN(basicColors.green)) {
        basicColors.green = max;
      }
      if (isNaN(basicColors.violet)) {
        basicColors.violet = min;
      }
    } else {
      if (isNaN(basicColors.green)) {
        basicColors.green = min;
      }
      if (isNaN(basicColors.violet)) {
        basicColors.violet = max;
      }
    }

    const actualColors = [];

    const basicColorsArray = makeColorsArray(basicColors);

    for (let k = 0; k < basicColorsArray.length; ++k) {
      if (!isNaN(basicColorsArray[k])) {
        actualColors.push(k);
      }
    }

    let i = 0;
    let j = 0;

    const fullColors = {};
    while (i < basicColorsArray.length || j < actualColors.length) {
      if (i < actualColors[j]) {
        const curIndex = Math.trunc(actualColors[j]);
        const prevIndex = Math.trunc(actualColors[j - 1]);
        const curValue = basicColorsArray[curIndex];
        const prevValue = basicColorsArray[prevIndex];
        // eslint-disable-next-line max-len
        const absentValue = (((curValue - prevValue) / (curIndex - prevIndex)) * (i - prevIndex)) + prevValue;
        fullColors[colorNames[i]] = absentValue;
      } else if (i === actualColors[j]) {
        fullColors[colorNames[i]] = basicColorsArray[i];
        j++;
      } else {
        throw new Error('we cannot reach there');
      }
      ++i;
    }
    return fullColors;
  }
}

export default ChartHeatmap;
