/* eslint-disable no-param-reassign,max-len */
import values from 'lodash/values';
import entries from 'lodash/entries';
import Mathx from '../../utils/Mathx';
import toDouble from '../common/toDouble';
import { escapeHtml, unquote } from '../../pages/old/utils/TemplateUtils';
import TimeFormatUtils from '../../utils/time/TimeFormatUtils';
import BinarySearch from './BinarySearch';

const hoverTemplate = (params) => `
<div class="hover${params.embed ? ' hover-embed' : ''}" style="display: none">
    <table class="hover-body">
        <tr>
            <td></td>
            <td class="hover-selected-time"></td>
            <td></td>
            ${params.norm ? '<td></td>' : ''}
        </tr>
        ${params.seriessForHover.map((s) => `
            <tr data-id="${unquote(s.salmonId)}" ${s.visible ? '' : 'style="display: none" class="hidden-from-hover"'}>
                <td>
                    <div class="solomon-color-square">
                        <div style="background-color: ${s.color}"></div>
                    </div>
                </td>
                <td class="hover-label">
                    ${escapeHtml(s.label)}
                </td>
                ${params.norm ? `
                    <td class="hover-number-value">
                        <div class="hover-value-max" style="visibility: hidden; height: 0"></div>
                        <div class="hover-value-norm-sel">-</div>
                    </td>` : ''}
                <td class="hover-number-value">
                    <div class="hover-value-max" style="visibility: hidden; height: 0"></div>
                    <div class="hover-value-sel">-</div>
                </td>
            </tr>
        `).join('')}
        ${params.seriessForHoverEllipsis ? `
            <tr>
                <td></td>
                <td>...</td>
                <td></td>
            </tr>` : ''}
        <tr class="hover-stack-sum hover-row-with-top-divider">
            <td></td>
            <td class="hover-label">
                Stack
            </td>
            ${params.norm ? '<td></td>' : ''}
            <td class="hover-number-value">
                <div class="hover-value-max" style="visibility: hidden; height: 0"></div>
                <div class="hover-value-sel"></div>
            </td>
        </tr>
    </table>
</div>`;

const MOSCOW_TZ_OFFSET = 180;
const MAX_DISTANCE_TO_HIGHLIGHT_SINGLE_POINT = 5;
const MAX_IN_HOVER = 11;

function alignNumber(num) {
  return num < 10 ? `0${num}` : `${num}`;
}

class GraphModeLinesHover {
  static renderHoverLines(
    $hoverPlaceholder,
    lines,
    norm,
    embed,
  ) {
    const params = {};

    const seriessForHover = lines.filter((seriess) => seriess.show);

    seriessForHover.sort((a1, a2) => {
      const a1val = a1.salmonStats.avg;
      const a2val = a2.salmonStats.avg;
      return Mathx.compareNanLow(a2val, a1val);
    });

    for (let i = 0; i < seriessForHover.length; ++i) {
      seriessForHover[i].visible = i < MAX_IN_HOVER;
    }

    params.seriessForHover = seriessForHover;
    params.embed = embed;
    params.seriessForHoverEllipsis = seriessForHover.length > MAX_IN_HOVER;
    params.norm = norm;

    $hoverPlaceholder.innerHTML = hoverTemplate(params);
  }

  static anyLineHasLineData(graphData) {
    for (let i = 0; i < graphData.length; i++) {
      if (graphData[i].rawData.length !== 0) {
        return true;
      }
    }

    return false;
  }

  static killHover($hoverPlaceholder, chartLines) {
    const $hover = $hoverPlaceholder.querySelector('.hover');
    if ($hover) {
      $hover.style.display = 'none';
    }

    chartLines.killHighlight();

    GraphModeLinesHover.disableHighlightAllItemsInHover($hoverPlaceholder);
  }

  static moveHoverDiv($hoverPlaceholder, $graph, cursorX, chartAreaLeftInsideGraph) {
    const $legendHover = $hoverPlaceholder.querySelector('.hover');

    if (!$legendHover) {
      return;
    }

    $legendHover.style.display = 'block';
    $legendHover.style.position = 'absolute';

    const top = $graph.offsetTop + 10;

    let left;
    const chartAreaOffsetLeft = $graph.offsetLeft + chartAreaLeftInsideGraph;
    if ($legendHover.clientWidth < cursorX - 10) {
      left = (chartAreaOffsetLeft + cursorX) - $legendHover.clientWidth - 10;
    } else {
      left = chartAreaOffsetLeft;
    }

    $legendHover.style.top = `${top}px`;
    $legendHover.style.left = `${left}px`;
  }

  /**
   * @param {Element} $hoverPlaceholder
   * @param {GraphFormatter} graphFormatter,
   * @param {ChartLines} chartLines,
   * @param {Object[]} graphData,
   * @param {boolean} norm,
   * @param {interpolate} interpolate,
   * @param {number} dataX
   */
  static updateHoverAndLegendWithCursor(
    $hoverPlaceholder,
    graphFormatter,
    chartLines,
    graphData,
    norm,
    interpolate,
    dataX,
    selectedIds,
  ) {
    const closestPoints = [];
    const selectedValueForLegendBySalmonId = {};

    let hasStackSeries = false;
    let stackSum = NaN;

    const rowBySalmonId = GraphModeLinesHover.constructRowsBySalmonId($hoverPlaceholder);

    const valueBySalmonId = {};

    for (let j = 0; j < graphData.length; ++j) {
      const seriesWithMetadata = graphData[j];

      const { salmonId } = seriesWithMetadata;

      const closestPoint = GraphModeLinesHover.findClosestPointOrNull(
        seriesWithMetadata.rawData,
        (point) => point[0],
        (point) => toDouble(point[1]),
        dataX,
        interpolate,
        chartLines,
      );

      const value = closestPoint ? toDouble(closestPoint[1]) : NaN;

      valueBySalmonId[salmonId] = value;

      if (seriesWithMetadata.area) {
        hasStackSeries = true;

        if (!isNaN(value)) {
          if (isNaN(stackSum)) {
            stackSum = value;
          } else {
            stackSum += value;
          }
        }
      }

      selectedValueForLegendBySalmonId[salmonId] = graphFormatter.format(value);

      if (!isNaN(value) && closestPoint) {
        const closestPointOrStrip = GraphModeLinesHover.findClosestPointOrNull(
          seriesWithMetadata.data,
          (point) => point[0],
          (point) => toDouble(point[1]),
          dataX,
          'linear',
          chartLines,
        );

        if (closestPointOrStrip) {
          closestPoints.push({
            salmonId,
            position: seriesWithMetadata.yaxisConf.positionValue,
            color: seriesWithMetadata.color,
            point: closestPointOrStrip,
          });
        }
      }
    }

    GraphModeLinesHover.updateSensorHoverValues('.hover-value-sel', rowBySalmonId, selectedValueForLegendBySalmonId);

    if (norm) {
      if (hasStackSeries) {
        GraphModeLinesHover.updateStackNormalizedHoverValues(rowBySalmonId, valueBySalmonId);
      } else {
        GraphModeLinesHover.updateLineNormalizedHoverValues(rowBySalmonId, valueBySalmonId);
      }
    }

    const selectedDate = new Date(dataX * 1000);
    const $selectedTime = $hoverPlaceholder.querySelector('.hover-selected-time');
    const localTzOrEmpty = GraphModeLinesHover.getLocalTimezone();
    const localTzPrefix = !localTzOrEmpty ? '' : ` (${localTzOrEmpty})`;
    $selectedTime.innerText = TimeFormatUtils.dateToDateHhMmBrowserTz(selectedDate) + localTzPrefix;

    const $stackSum = $hoverPlaceholder.querySelector('.hover-stack-sum');
    if (hasStackSeries && graphData.length > 1) {
      const stackSumFormatted = graphFormatter.format(stackSum);
      const $hoverValueSel = $stackSum.querySelector('.hover-value-sel');
      $hoverValueSel.innerText = stackSumFormatted;
      $stackSum.style.display = 'table-row';
    } else {
      $stackSum.style.display = 'none';
    }

    chartLines.highlightPoints(closestPoints);

    // eslint-disable-next-line max-len
    GraphModeLinesHover.highlightItemsBySalmonIds($hoverPlaceholder, selectedIds, rowBySalmonId);
  }

  static updateStackNormalizedHoverValues(
    rowBySalmonId,
    valueBySalmonId,
  ) {
    const sum = values(valueBySalmonId)
      .filter((v) => !isNaN(v) && Number.isFinite(v))
      .map((v) => Math.abs(v))
      .reduce((a, b) => a + b, 0);

    const normValueBySalmonId = {};

    const salmonIdAndValues = entries(valueBySalmonId);

    for (let i = 0; i < salmonIdAndValues.length; ++i) {
      const entry = salmonIdAndValues[i];
      const salmonId = entry[0];
      const value = entry[1];

      let normValue;

      if (sum === 0) {
        normValue = '';
      } else if (isNaN(value)) {
        normValue = '0.0%';
      } else {
        const num = (100 * Math.abs(value)) / sum;
        normValue = `${num.toFixed(1)}%`;
      }

      normValueBySalmonId[salmonId] = normValue;
    }

    GraphModeLinesHover.updateSensorHoverValues('.hover-value-norm-sel', rowBySalmonId, normValueBySalmonId);
  }

  static updateLineNormalizedHoverValues(
    rowBySalmonId,
    valueBySalmonId,
  ) {
    const filteredValues = values(valueBySalmonId)
      .filter((v) => !isNaN(v) && Number.isFinite(v))
      .map((v) => Math.abs(v));

    const max = filteredValues.length === 0
      ? 0
      : filteredValues.reduce((a, b) => Math.max(a, b));

    const normValueBySalmonId = {};

    const salmonIdAndValues = entries(valueBySalmonId);

    for (let i = 0; i < salmonIdAndValues.length; ++i) {
      const entry = salmonIdAndValues[i];
      const salmonId = entry[0];
      const value = entry[1];

      let normValue;

      if (max === 0) {
        normValue = '';
      } else if (isNaN(value)) {
        normValue = '0.0%';
      } else {
        const number = (100 * Math.abs(value)) / max;
        normValue = `${number.toFixed(1)}%`;
      }

      normValueBySalmonId[salmonId] = normValue;
    }

    GraphModeLinesHover.updateSensorHoverValues('.hover-value-norm-sel', rowBySalmonId, normValueBySalmonId);
  }

  static getLocalTimezone() {
    let tzOffset = -new Date().getTimezoneOffset();

    if (tzOffset === MOSCOW_TZ_OFFSET) {
      return '';
    }

    if (tzOffset === 0) {
      return 'UTC';
    }

    tzOffset -= MOSCOW_TZ_OFFSET;

    const sign = tzOffset < 0 ? '-' : '+';

    tzOffset = Math.abs(tzOffset);

    const hours = tzOffset / 60;
    const minutes = tzOffset % 60;

    if (minutes === 0) {
      return `MSK${sign}${hours}`;
    }

    return `MSK${sign}${hours}:${alignNumber(minutes)}`;
  }

  static constructRowsBySalmonId($hoverPlaceholder) {
    const $elements = $hoverPlaceholder.querySelectorAll('[data-id]');
    const rowBySalmonId = {};
    for (let i = 0; i < $elements.length; ++i) {
      const $element = $elements[i];
      const salmonId = $element.getAttribute('data-id');
      rowBySalmonId[salmonId] = $element;
    }
    return rowBySalmonId;
  }

  static updateSensorHoverValues(
    selector,
    rowBySalmonId,
    selectedValueBySalmonId,
  ) {
    const salmonIdAndValues = entries(selectedValueBySalmonId);
    for (let i = 0; i < salmonIdAndValues.length; ++i) {
      const pair = salmonIdAndValues[i];
      const salmonId = pair[0];
      const value = pair[1];
      const $row = rowBySalmonId[salmonId];
      if ($row) {
        const $el = $row.querySelector(selector);
        if ($el) {
          $el.innerText = value;
        }
      }
    }
  }

  static highlightItemsBySalmonIds(
    $hoverPlaceholder,
    highlightedSalmonIds,
    rowBySalmonId,
  ) {
    GraphModeLinesHover.disableHighlightAllItemsInHover($hoverPlaceholder);
    for (let i = 0; i < highlightedSalmonIds.length; ++i) {
      const salmonId = highlightedSalmonIds[i];
      const $row = rowBySalmonId[salmonId];
      if ($row) {
        $row.style.fontWeight = 'bold';
        if ($row.classList.contains('hidden-from-hover')) {
          $row.style.display = 'table-row';
        }
      }
    }
  }

  static findClosestPointOrNull(
    points,
    xExtractor,
    yExtractor,
    target,
    interpolate,
    chartLines,
  ) {
    if (points.length === 0) {
      return null;
    }

    const targetIndex = GraphModeLinesHover.lowerBound(points, xExtractor, target);

    if (targetIndex === 0) {
      const firstPoint = points[0];
      return GraphModeLinesHover.returnSinglePointOrNull(firstPoint, xExtractor, target, chartLines);
    } if (targetIndex === points.length) {
      const lastPoint = points[points.length - 1];
      return GraphModeLinesHover.returnSinglePointOrNull(lastPoint, xExtractor, target, chartLines);
    }
    const point = points[targetIndex];
    const prevPoint = points[targetIndex - 1];
    switch (interpolate) {
      case 'linear': {
        const closestPoint = GraphModeLinesHover.chooseClosestPoint(point, prevPoint, target, xExtractor);
        if (GraphModeLinesHover.isNoDataInterval(prevPoint, point, yExtractor)) {
          return GraphModeLinesHover.returnSinglePointOrNull(closestPoint, xExtractor, target, chartLines);
        }
        return closestPoint;
      }
      case 'none': {
        const closestPoint = GraphModeLinesHover.chooseClosestPoint(point, prevPoint, target, xExtractor);
        return GraphModeLinesHover.returnSinglePointOrNull(closestPoint, xExtractor, target, chartLines);
      }
      case 'left':
        return prevPoint;
      case 'right':
        return point;
      default:
        throw new Error(`Unknown interpolation: ${interpolate}`);
    }
  }

  static isNoDataInterval(prevPoint, point, yExtractor) {
    return isNaN(yExtractor(prevPoint)) || isNaN(yExtractor(point));
  }

  static returnSinglePointOrNull(
    point,
    xExtractor,
    target,
    chartLines,
  ) {
    const pointDataX = xExtractor(point);
    const chartX1 = chartLines.geom.dataXToChartAreaX(target);
    const chartX2 = chartLines.geom.dataXToChartAreaX(pointDataX);
    if (Math.abs(chartX1 - chartX2) <= MAX_DISTANCE_TO_HIGHLIGHT_SINGLE_POINT) {
      return point;
    }
    return null;
  }

  static lowerBound(
    points,
    extractor,
    target,
  ) {
    return BinarySearch.lowerBound(
      points.length,
      (index) => extractor(points[index]),
      target,
    );
  }

  static chooseClosestPoint(point, prevPoint, dataX, xExtractor) {
    const prevX = xExtractor(prevPoint);
    const thisX = xExtractor(point);

    if (dataX - prevX < thisX - dataX) {
      return prevPoint;
    }
    return point;
  }

  static disableHighlightAllItemsInHover($hoverPlaceholder) {
    const $elements = $hoverPlaceholder.querySelectorAll('[data-id]');

    for (let i = 0; i < $elements.length; ++i) {
      const $element = $elements[i];

      $element.style.fontWeight = 'normal';
      if ($element.classList.contains('hidden-from-hover')) {
        $element.style.display = 'none';
      }
    }
  }
}

export default GraphModeLinesHover;
