/* eslint-disable no-throw-literal,max-len,no-plusplus,no-param-reassign,guard-for-in,no-restricted-syntax */
import SvgUtils from '../svg/SvgUtils';
import YaxisCommon from '../common/YaxisCommon';
import ChartCommon from '../common/ChartCommon';
import YAxisScaleData from '../common/YAxisScaleData';
import SvgDBuilder from '../common/SvgDBuilder';
import toDouble from '../common/toDouble';
import CursorHover from '../hover/CursorHover';
import { removeElementsBySelector } from '../common/SelectorUtils';

function isPrimaryButtonPressed(event) {
  return event.button === 0;
}

function getSeconds(point) {
  return point[0];
}

function getYHigh(point) {
  return toDouble(point[1]);
}

function getYLow(point) {
  return toDouble(point[2]);
}

class ChartLines {
  /**
   * @param {Element} $chartArea
   * @param {ChartGeom} geom
   */
  constructor($chartArea, geom) {
    this.$chartArea = $chartArea;
    this.geom = geom;
  }

  /**
   *
   * @param {PointOrStrip[]} points
   * @return {PointOrStrip[][]}
   */
  static splitPointsBySegments(points) {
    const r = [];

    let index = 0;
    while (index < points.length) {
      const segmentStart = ChartLines.findNonZeroSegmentStart(points, index);

      if (index < segmentStart) {
        // Append full zero segment
        const zeroSegment = [];
        for (let i = index; i <= segmentStart; ++i) {
          zeroSegment.push(points[i]);
        }
        r.push(zeroSegment);
      }

      const current = [];

      const segmentEnd = ChartLines.findNonZeroSegmentEnd(points, segmentStart);

      if (segmentStart === segmentEnd && segmentStart !== 0) {
        break;
      }

      for (let i = segmentStart; i <= segmentEnd; ++i) {
        current.push(points[i]);
      }

      if (current.length > 0) {
        r.push(current);
      }

      if (segmentEnd === points.length - 1) {
        // all areas were painted
        break;
      }

      index = segmentEnd;
    }

    return r;
  }

  /**
   * @param {PointOrStrip[]} segment
   * @return {boolean}
   */
  static isZeroHeightArea(segment) {
    for (let index = 0; index < segment.length; ++index) {
      const point = segment[index];
      if (getYHigh(point) !== getYLow(point)) {
        return false;
      }
    }
    return true;
  }

  /**
   *
   * @param {PointOrStrip[]} segment
   * @param {number} fromIndex
   * @return {number}
   */
  static findNextHighPoint(segment, fromIndex) {
    let index = fromIndex + 1;

    // Look at all intermediate points
    for (; index < segment.length - 1; index++) {
      const prevY = getYHigh(segment[index - 1]);
      const curY = getYHigh(segment[index]);
      const nextY = getYHigh(segment[index + 1]);

      // ignore points that are equal to neighbours
      if (!(prevY === curY && curY === nextY)) {
        break;
      }
    }

    return index;
  }

  /**
   * @param {PointOrStrip[]} segment
   * @param {number} fromIndex
   * @return {number}
   */
  static findNextLowPoint(segment, fromIndex) {
    let index = fromIndex - 1;

    // look at all intermediate points in back direction
    for (; index > 0; index--) {
      const prevY = getYLow(segment[index + 1]);
      const curY = getYLow(segment[index]);
      const nextY = getYLow(segment[index - 1]);

      // ignore points that are equal to neighbours
      if (!(prevY === curY && curY === nextY)) {
        break;
      }
    }

    return index;
  }

  /**
   *
   * @param {PointOrStrip[]} segment
   * @param {number} fromIndex
   * @return {number}
   */
  static findNonZeroSegmentEnd(segment, fromIndex) {
    let index = fromIndex;
    while (index + 1 < segment.length) {
      index++;
      let point = segment[index];
      // inc index first, cause we have to draw the first point with YHigh == YLow
      if (getYLow(point) === getYHigh(point)) {
        // in a simple case we can write
        // break;
        // right there, but
        // there is benefit of splitting the area if only we have two points in a row with equal YHigh == YLow.
        // let's try to look forward
        const nextIndex = index + 1;
        if (nextIndex >= segment.length) {
          break;
        }
        point = segment[nextIndex];
        if (getYLow(point) === getYHigh(point)) {
          break;
        }
      }
    }
    return index;
  }

  /**
   * @param {PointOrStrip[]} segment
   * @param {number} fromIndex
   * @return {number}
   */
  static findNonZeroSegmentStart(segment, fromIndex) {
    while (fromIndex < segment.length) {
      const point = segment[fromIndex];
      if (getYLow(point) === getYHigh(point)) {
        fromIndex++;
      } else {
        // we should include one point with YLow==YHigh (if it exists)
        // but if index == 0, we can't do it
        return fromIndex === 0 ? 0 : fromIndex - 1;
      }
    }
    return segment.length - 1;
  }

  /**
   * @param {YAxisScaleData} lines
   * @param {ChartConf} conf
   * @return {YAxisScaleData}
   */
  static computeLinesYAxisMinMax(lines, conf) {
    const datamin = conf.yaxis.min === 'datamin';
    const allPoints = ChartLines.computeAllYs(lines, datamin);

    let r;
    if (allPoints.length === 0) {
      r = new YAxisScaleData(0, 0, 0, 0);
    } else {
      const min = allPoints.reduce((a, b) => Math.min(a, b));
      const max = allPoints.reduce((a, b) => Math.max(a, b));

      // For log scale we want to calculate a member with smallest absolute value
      // If all sequence members are zero then and only then it should be zero.
      // Since we only need this for log scale, we may drop this calculation in any other case.
      let minAbs = allPoints.reduce((a, b) => {
        if (a === 0) {
          return Math.abs(b);
        }
        if (b === 0) {
          return Math.abs(a);
        }
        return Math.min(Math.abs(a), Math.abs(b));
      });
      minAbs = Math.abs(minAbs); // for 1-element array

      // We want to calculate a member with smallest value
      const minValue = min;

      r = new YAxisScaleData(min, max, minAbs, minValue);
    }

    return ChartCommon.adjustMinMaxWithConf(r, conf.yaxis);
  }

  /**
   * @param {LineData[]} lines
   * @param {boolean} datamin
   * @return {number[]}
   */
  static computeAllYs(lines, datamin) {
    const r = [];

    for (let i = 0; i < lines.length; ++i) {
      const line = lines[i];
      for (let j = 0; j < line.data.length; ++j) {
        const pointOrStrip = line.data[j];
        if (datamin) {
          const yHigh = getYHigh(pointOrStrip);
          const yLow = getYLow(pointOrStrip);
          if (!isNaN((yHigh)) && (isNaN(yLow) || yHigh >= yLow)) {
            r.push(yHigh);
          } else if (!isNaN(yLow)) {
            r.push(yLow);
          }
        } else {
          // zero element is timestamp
          for (let k = 1; k < pointOrStrip.length; ++k) {
            const y = toDouble(pointOrStrip[k]);
            if (!isNaN(y)) {
              r.push(y);
            }
          }
        }
      }
    }

    return r;
  }

  /**
   * @param {LineData[]} lines
   * @param {ChartConf[]} conf
   * @return {YAxisScaleData}
   */
  static computeLinesYAxisMinMaxPlusSome(lines, conf) {
    const yScaleData = ChartLines.computeLinesYAxisMinMax(lines, conf);
    return ChartLines.addSomeIfNeeded(yScaleData, conf);
  }

  /**
   * @param {YAxisScaleData} yScaleData
   * @param {ChartConf} conf
   * @return {YAxisScaleData}
   */
  static addSomeIfNeeded(yScaleData, conf) {
    const yaxisMin = yScaleData.getMin();
    const yaxisMax = yScaleData.getMax();

    const configMin = conf.yaxis.min;
    const configMax = conf.yaxis.max;

    const hasConfigMinValue = configMin != null && !!configMin
      && configMin !== 'datamin';

    const hasConfigMaxValue = configMax != null && !!configMax;

    if (yaxisMin === 0 && yaxisMax === 0 && !hasConfigMinValue && !hasConfigMaxValue) {
      return new YAxisScaleData(-1, 1, 0, 0);
    }
    if (yaxisMin === yaxisMax) {
      const value = yaxisMin;
      const sign = Math.sign(value);
      const valueAbs = Math.abs(value);
      return new YAxisScaleData((valueAbs / 2) * sign, (valueAbs / 2) * 3 * sign, valueAbs / 2, value);
    }
    const some = 0.05 * (yaxisMax - yaxisMin);

    const showMinStrictly = hasConfigMinValue || yaxisMin === 0;
    const showMaxStrictly = hasConfigMaxValue || yaxisMax === 0;

    let newYAxisMin;
    if (showMinStrictly) {
      newYAxisMin = yaxisMin;
    } else {
      newYAxisMin = yaxisMin - some;
      if (newYAxisMin < 0 && yaxisMin > 0) {
        newYAxisMin = 0;
      }
    }

    let newYaxisMax;
    if (showMaxStrictly) {
      newYaxisMax = yaxisMax;
    } else {
      newYaxisMax = yaxisMax + some;
    }
    return new YAxisScaleData(newYAxisMin, newYaxisMax, yScaleData.getMinAbs(), yScaleData.getMinValue());
  }

  /**
   * @param {PointOrStrip[]} points
   * @return {PointOrStrip[][]}
   */
  static splitPointsByGaps(points) {
    const r = [];
    let current = [];

    for (let i = 0; i < points.length; ++i) {
      const point = points[i];
      if (!isNaN(getYHigh(point))) {
        current.push(point);
      } else if (current.length > 0) {
        r.push(current);
        current = [];
      }
    }

    if (current.length > 0) {
      r.push(current);
    }

    return r;
  }

  /**
   * @param {ChartConf} conf
   * @param {SVGElement} $xaxisArea
   * @param {SVGElement} $chartArea
   * @param {ChartGeom} geom
   */
  static drawXAxisLabels(conf, $xaxisArea, $chartArea, geom) {
    const { ticks } = conf.xaxis;

    for (let i = 0; i < conf.xaxis.ticks.length; ++i) {
      const tick = ticks[i];

      const chartX = geom.dataXToChartAreaX(tick[0]);
      const chartXForText = chartX + geom.chartBorderWidth;

      const $text = SvgUtils.createTextElement();
      $text.setAttribute('class', 'chart-axis-label');
      $text.setAttribute('x', `${chartXForText}`);
      $text.setAttribute('dy', '1em');
      $text.setAttribute('text-anchor', 'middle');
      SvgUtils.appendText($text, tick[1]);
      $xaxisArea.appendChild($text);

      const d = new SvgDBuilder();
      d.moveSharp(chartX, 0, 1);
      d.lineV(10000);

      const $path = SvgUtils.createPathElementWithD('chart-grid', d);
      $chartArea.appendChild($path);
    }
  }

  /**
   * @param {ChartConf} conf
   * @param {ChartGeom} geom
   * @param {SVGElement} $chartArea
   */
  static drawWeekends(conf, geom, $chartArea) {
    const { weekends } = conf.xaxis;
    for (let i = 0; i < weekends.length; ++i) {
      const weekend = weekends[i];
      const chartBegin = geom.dataXToChartAreaX(weekend.begin);
      const chartEnd = geom.dataXToChartAreaX(weekend.end);

      const d = new SvgDBuilder();
      d.moveSharp(chartBegin, 0, 1);
      d.lineV(1000);
      d.lineH(chartEnd);
      d.lineV(0);

      const $path = SvgUtils.createPathElement();

      $path.setAttribute('fill', 'beige');
      $path.setAttribute('d', d.format());
      $chartArea.appendChild($path);
    }
  }

  /**
   * @param {ChartGeom} geom
   * @param {SVGElement} $chartArea
   * @param {ChartConf} conf
   */
  static drawGridX(geom, $chartArea, conf) {
    const { ticks } = conf.xaxis;

    for (let i = 0; i < ticks.length; ++i) {
      const tick = ticks[i];
      const chartAreaX = geom.dataXToChartAreaX(tick[0]);

      const d = new SvgDBuilder();
      d.moveSharp(chartAreaX, 0, 1);
      d.lineV(10000);

      const $path = SvgUtils.createPathElementWithD('chart-grid', d);
      $chartArea.appendChild($path);
    }
  }

  /**
   *
   * @param {SVGElement} $chartArea
   * @param {ChartGeom} geom
   * @param {ChartConf} conf
   */
  static drawNowVline($chartArea, geom, conf) {
    const chartX = geom.dataXToChartAreaX(conf.xaxis.now);

    const d = new SvgDBuilder();
    d.moveSharp(chartX, 0, 1);
    d.lineV(10000);
    const $path = SvgUtils.createPathElementWithD('chart-now', d);
    $chartArea.appendChild($path);
  }

  /**
   * @param {SVGElement} $chartArea
   * @param {ChartGeom} geom
   * @param {number} threshold
   */
  static drawThreshold($chartArea, geom, threshold) {
    const lineY = geom.dataYToChartAreaY('left', threshold);

    const d = new SvgDBuilder();
    d.moveSharp(0, lineY, 1);
    d.lineSharp(geom.chartAreaWidth, lineY, 2);

    const $path = SvgUtils.createPathElement();
    $path.setAttribute('d', d.format());
    $path.setAttribute('stroke', 'red');
    $path.setAttribute('stroke-width', '2');
    $chartArea.appendChild($path);
  }

  /**
   * Draw chart with lines and areas
   *
   * @param {Element} $where
   * @param {LineData[]} lines
   * @param {ChartConf} conf
   * @param {number} chartBorderWidth
   * @param {boolean} hasMinYaxisAreaWidth
   * @param {string} borderColor
   * @return {ChartLines}
   */
  static chart(
    $where,
    lines,
    conf,
    chartBorderWidth,
    hasMinYaxisAreaWidth,
    borderColor,
  ) {
    const groupedByYaxisPosition = YaxisCommon.groupLinesByYaxisPosition(lines);

    const axisScaleDataMap = {};

    for (const yaxisName in groupedByYaxisPosition) {
      const yaxisLines = groupedByYaxisPosition[yaxisName];
      axisScaleDataMap[yaxisName] = ChartLines.computeLinesYAxisMinMaxPlusSome(yaxisLines, conf);
    }

    const xaxisRange = conf.xaxis.max - conf.xaxis.min;

    const common = ChartCommon.makeCommonChartElements(
      $where, axisScaleDataMap, xaxisRange, conf, chartBorderWidth, hasMinYaxisAreaWidth,
    );

    const { geom } = common;

    ChartLines.drawXAxisLabels(conf, common.$xaxisArea, common.$chartArea, geom);

    YaxisCommon.drawYAxisLabels(conf, common);

    ChartLines.drawWeekends(conf, geom, common.$chartArea);

    ChartLines.drawData(common.$chartArea, geom, groupedByYaxisPosition);

    YaxisCommon.drawGridY(geom, common.$chartArea);

    ChartLines.drawGridX(geom, common.$chartArea, conf);
    ChartLines.drawNowVline(common.$chartArea, geom, conf);
    ChartCommon.drawChartAreaBorder(common.$chartAreaWithBorder, geom, borderColor);

    const { threshold } = conf;
    if (!isNaN(threshold) && threshold !== null) {
      ChartLines.drawThreshold(common.$chartArea, geom, threshold);
    }

    const chartLines = new ChartLines(common.$chartArea, geom);

    chartLines.setupInteractivity();

    return chartLines;
  }

  static drawData($chartArea, geom, linesGroupedByYaxisPosition) {
    // we should draw at first areas and then lines to make lines visible (An area hide a line if the line is under the area)
    ChartLines.drawDataByYaxisPosition($chartArea, geom, linesGroupedByYaxisPosition, 'left');
    ChartLines.drawDataByYaxisPosition($chartArea, geom, linesGroupedByYaxisPosition, 'right');
  }

  static drawDataByYaxisPosition($chartArea, geom, linesGroupedByYaxisPosition, position) {
    const lines = linesGroupedByYaxisPosition[position];
    if (lines) {
      for (let i = 0; i < lines.length; ++i) {
        ChartLines.drawLine($chartArea, geom, lines[i]);
      }
    }
  }

  static drawLine($chartArea, geom, line) {
    const segments = ChartLines.splitPointsByGaps(line.data);
    for (let j = 0; j < segments.length; ++j) {
      ChartLines.drawLineSegment($chartArea, geom, segments[j], line);
    }
  }

  /**
   * @param {Element} $chartArea
   * @param {ChartGeom} geom
   * @param {PointOrStrip[]} segment
   * @param {LineData} line
   */
  static drawLineSegment($chartArea, geom, segment, line) {
    const singlePoint = segment.length === 1 && !line.area;

    if (line.area) {
      ChartLines.drawArea($chartArea, geom, segment, line);
    } else {
      ChartLines.drawPolyLine($chartArea, geom, segment, line, singlePoint);
    }
  }

  /**
   * @param {PointOrStrip[]} segment
   * @param {LineData} line
   * @param {boolean} singlePoint
   */
  static drawPolyLine($chartArea, geom, segment, line, singlePoint) {
    const d = new SvgDBuilder();

    const position = line.yaxisConf.positionValue;
    const { color } = line;
    for (let index = 0; index < segment.length; ++index) {
      const point = segment[index];
      const chartAreaX = geom.dataXToChartAreaX(getSeconds(point));
      const chartAreaY = geom.dataYToChartAreaY(position, getYHigh(point));
      if (index === 0) {
        d.move(chartAreaX, chartAreaY);

        if (singlePoint) {
          d.loop();
        }
      } else {
        d.line(chartAreaX, chartAreaY);
      }
    }

    const $path = SvgUtils.createPathElement();
    if (singlePoint) {
      $path.setAttribute('stroke-linecap', 'round');
      $path.setAttribute('fill', 'none');
      $path.setAttribute('stroke', color);
      $path.setAttribute('stroke-width', '3px');
    } else {
      $path.setAttribute('stroke-width', '2px');
      $path.setAttribute('fill', 'none');
      $path.setAttribute('stroke', color);
      $path.setAttribute('pointer-events', 'visibleStroke');
    }

    $path.setAttribute('d', d.format());
    $path.setAttribute('salmon-id', line.salmonId);

    $chartArea.appendChild($path);
  }

  /**
   * @param {PointOrStrip[]} points
   * @param {LineData} line
   */
  static drawArea($chartArea, geom, points, line) {
    const segments = ChartLines.splitPointsBySegments(points);

    const position = line.yaxisConf.positionValue;
    const { color } = line;
    for (let i = 0; i < segments.length; i++) {
      const segment = segments[i];

      const d = ChartLines.makeDForNonZeroAreaZegment(geom, segment, position);

      const $path = SvgUtils.createPathElement();
      if (segment.length === 1) {
        $path.setAttribute('stroke-width', '2px');
        $path.setAttribute('fill', 'none');
        $path.setAttribute('stroke', color);
      } else {
        // area chart, but not if single pair
        $path.setAttribute('stroke', 'none');
        $path.setAttribute('fill', color);
      }
      $path.setAttribute('d', d.format());
      $path.setAttribute('salmon-id', line.salmonId);

      $chartArea.appendChild($path);

      const isZeroHeightArea = ChartLines.isZeroHeightArea(segment);
      const isZeroArea = ChartLines.isZeroArea(geom, segment, position);

      const isZeroHeightAreaAboveOtherAreas = isZeroHeightArea && !isZeroArea;

      if (segment.length > 1 && !isZeroHeightAreaAboveOtherAreas) {
        const $highPath = ChartLines.drawHighLine(geom, segment, position, color);
        $chartArea.appendChild($highPath);
      }
    }
  }

  static drawHighLine(geom, segment, position, color) {
    const d = new SvgDBuilder();
    ChartLines.drawYHighValues(geom, segment, position, d);

    const $path = SvgUtils.createPathElement();
    $path.setAttribute('fill', 'none');
    $path.setAttribute('stroke-width', '1.5px');

    if (ChartLines.isZeroArea(geom, segment, position)) {
      $path.setAttribute('stroke', color);
    } else {
      $path.setAttribute('stroke', 'black');
      $path.setAttribute('stroke-opacity', '0.2');
    }
    $path.setAttribute('pointer-events', 'none');

    $path.setAttribute('d', d.format());

    return $path;
  }

  /**
   * @param {PointOrStrip[]} segment
   * @param {string} position
   */
  static isZeroArea(geom, segment, position) {
    const yaxis = geom.getYaxis(position);
    const ymin = yaxis ? 0 : yaxis.minValue;
    for (let i = 0; i < segment.length; i++) {
      if (getYHigh(segment[i]) !== ymin) {
        return false;
      }
    }
    return true;
  }

  static makeDForNonZeroAreaZegment(geom, segment, position) {
    const d = new SvgDBuilder();

    // go around the perimeter:
    // at first we draw YHigh values, then in back direction YLow values
    ChartLines.drawYHighValues(geom, segment, position, d);
    ChartLines.drawYLowValuesInBackDirection(geom, segment, position, d);

    d.loop();

    return d;
  }

  static drawYHighValues(geom, segment, position, d) {
    // Drawing start point and next points, ignoring intermediate with same highY
    for (let index = 0; index < segment.length; index = ChartLines.findNextHighPoint(segment, index)) {
      const point = segment[index];
      const chartAreaX = geom.dataXToChartAreaX(getSeconds(point));
      const chartAreaY = geom.dataYToChartAreaY(position, getYHigh(point));
      if (index === 0) {
        d.move(chartAreaX, chartAreaY);
      } else {
        d.line(chartAreaX, chartAreaY);
      }
    }
  }

  static drawYLowValuesInBackDirection(geom, segment, position, d) {
    // Drawing start point and next points in back direction, ignoring intermediate with same lowY
    for (let index = segment.length - 1; index >= 0; index = ChartLines.findNextLowPoint(segment, index)) {
      const point = segment[index];
      const chartAreaX = geom.dataXToChartAreaX(getSeconds(point));
      const chartAreaY = geom.dataYToChartAreaY(position, getYLow(point));
      d.line(chartAreaX, chartAreaY);
    }
  }

  static computeSelectedAreaIds(event) {
    try {
      const target = event.target || null;
      if (!target || !(target instanceof Element)) {
        return [];
      }
      const id = target.getAttribute('salmon-id');
      if (!id) {
        return [];
      }
      return [id];
    } catch (e) {
      console.error(e);
      return [];
    }
  }

  setClickListener(clickListener) {
    this.clickListener = clickListener;
  }

  setSelectListener(selectListener) {
    this.selectListener = selectListener;
  }

  setLeaveListener(leaveListener) {
    this.leaveListener = leaveListener;
  }

  setHoverListener(hoverListener) {
    this.hoverListener = hoverListener;
  }

  clearCursor() {
    CursorHover.clearCursor(this.$chartArea);
    removeElementsBySelector(this.$chartArea, '.chart-v-select');
  }

  clearSelection() {
    this.clearCursor();
    this.selecting = NaN;
  }

  makeSelectTo(chartAreaX) {
    const d = new SvgDBuilder();
    d.moveSharp(this.selecting, 0, 1);
    d.lineV(10000);
    d.lineH(chartAreaX);
    d.lineV(0);

    const $path = SvgUtils.createPathElement();
    $path.setAttribute('class', 'chart-v-select');
    $path.setAttribute('d', d.format());
    this.$chartArea.appendChild($path);
  }

  fireChartClick(clientX) {
    const chartAreaX = clientX - this.geom.getChartAreaClientLeft();
    const dataX = this.geom.chartAreaXToDataX(chartAreaX);

    if (this.clickListener) {
      this.clickListener({ chartAreaX, dataX });
    }
  }

  fireClickOrSelect(event) {
    this.clearCursor();
    // TODO: small offset is OK too
    if (event.clientX - this.geom.getChartAreaClientLeft() === this.selecting) {
      this.fireChartClick(event.clientX);
      // eslint-disable-next-line no-restricted-globals
    } else if (!isNaN(this.selecting)) {
      const dataBegin = this.geom.chartAreaXToDataX(this.selecting);
      const dataEnd = this.geom.chartAreaXToDataX(event.clientX - this.geom.getChartAreaClientLeft());

      const dataRange = [dataBegin, dataEnd];

      if (this.selectListener) {
        this.selectListener(dataRange);
      }
    }
    this.selecting = NaN;
  }

  onMouseMove = (event) => {
    this.clearCursor();
    const selectedIds = ChartLines.computeSelectedAreaIds(event);
    const chartAreaX = event.clientX - this.geom.getChartAreaClientLeft();
    if (isNaN(this.selecting)) {
      CursorHover.drawCursor(this.$chartArea, chartAreaX);

      const chartAreaOffsetLeft = this.geom.chartAreaOffsetLeft + 2;
      const dataX = this.geom.chartAreaXToDataX(chartAreaX);
      const hoverEvent = {
        chartLines: this, chartAreaX, chartAreaOffsetLeft, dataX, selectedIds,
      };
      if (this.hoverListener) {
        this.hoverListener(hoverEvent);
      }
    } else {
      this.makeSelectTo(chartAreaX);
    }
  };

  onMouseLeave = (event) => {
    if (!isNaN(this.selecting)) {
      this.fireClickOrSelect(event);
    } else {
      this.clearCursor();
    }
    if (this.leaveListener) {
      this.leaveListener({ chartLines: this });
    }
  };

  onMouseDown = (event) => {
    if (!isPrimaryButtonPressed(event)) {
      return;
    }
    this.clearCursor();
    this.selecting = event.clientX - this.geom.getChartAreaClientLeft();
    this.makeSelectTo(this.selecting);
  };

  onMouseUp = (event) => {
    if (!isPrimaryButtonPressed(event) || isNaN(this.selecting)) {
      return;
    }
    this.fireClickOrSelect(event);
  };

  killHighlight() {
    CursorHover.clearPoints(this.$chartArea);
  }

  highlight(pointWithLineOptions) {
    const yHigh = getYHigh(pointWithLineOptions.point);
    if (!isNaN(yHigh)) {
      CursorHover.drawPoint(
        this.$chartArea,
        this.geom.dataXToChartAreaX(getSeconds(pointWithLineOptions.point)),
        this.geom.dataYToChartAreaY(pointWithLineOptions.position, yHigh),
        pointWithLineOptions.color,
      );
    }
  }

  highlightPoints(highlightedPoints) {
    this.killHighlight();
    for (let i = 0; i < highlightedPoints.length; ++i) {
      const pointWithLineOptions = highlightedPoints[i];
      this.highlight(pointWithLineOptions);
    }
  }

  setupInteractivity() {
    this.$chartArea.addEventListener('mousemove', this.onMouseMove);
    this.$chartArea.addEventListener('mouseleave', this.onMouseLeave);
    this.$chartArea.addEventListener('mousedown', this.onMouseDown);
    this.$chartArea.addEventListener('mouseup', this.onMouseUp);
  }

  destroy() {
    this.$chartArea.removeEventListener('mousemove', this.onMouseMove);
    this.$chartArea.removeEventListener('mouseleave', this.onMouseLeave);
    this.$chartArea.removeEventListener('mousedown', this.onMouseDown);
    this.$chartArea.removeEventListener('mouseup', this.onMouseUp);
  }
}

export default ChartLines;
