/* eslint-disable max-len */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import ReactChartLines from './ReactChartLines';
import * as XAxisConfUtils from '../xaxis/XAxisConfUtils';
import HealthAlerting from '../../pages/old/OldGraphPage/HealthAlerting';
import GraphConfUtils from '../../pages/old/OldGraphPage/utils/GraphConfUtils';
import GraphModeLinesHover from '../hover/GraphModeLinesHover';
import parseBoolean from '../../utils/boolean';
import BrowserUtils from '../../utils/BrowserUtils';
import TimeFormatUtils from '../../utils/time/TimeFormatUtils';
import UrlUtils from '../../utils/url/UrlUtils';
import UserLinksBasic from '../../utils/UserLinksBasic';

function graphClickToExit(locationHref) {
  const goTo = UrlUtils.removeQueryArgs(
    locationHref,
    UserLinksBasic.GRAPH_ONLY_QA,
    UserLinksBasic.LEGEND_QUERY_ARG,
  );

  try {
    window.top.location = goTo;
  } catch (e) {
    console.warn(`failed to change location to ${goTo}`, e);
  }
}

class GraphModeLines extends PureComponent {
  static goToLocal(beginSeconds, endSeconds) {
    const newB = TimeFormatUtils.formatUtcMillis(beginSeconds * 1000);
    const newE = TimeFormatUtils.formatUtcMillis(endSeconds * 1000);

    let newUrl = BrowserUtils.getLogicalUrlAfterHost();
    newUrl = UrlUtils.updateParameter(newUrl, UserLinksBasic.B_QA, newB);
    newUrl = UrlUtils.updateParameter(newUrl, UserLinksBasic.E_QA, newE);

    try {
      window.location = newUrl;
    } catch (e) {
      console.warn('failed to reload page', e);
    }
  }

  componentDidMount() {
    const norm = parseBoolean(this.props.graphConf.norm);
    const embed = this.props.graphOnly;
    GraphModeLinesHover.renderHoverLines(this._el._hover, this.props.lines, norm, embed);
  }

  onHover = ({
    chartLines, chartAreaX, chartAreaOffsetLeft, dataX, selectedIds,
  }) => {
    if (!GraphModeLinesHover.anyLineHasLineData(this.props.lines)) {
      return;
    }

    const interpolate = this.props.graphConf.interpolate || 'linear';
    const norm = parseBoolean(this.props.graphConf.norm);

    GraphModeLinesHover.updateHoverAndLegendWithCursor(
      this._el._hover,
      this.props.graphFormatter,
      chartLines,
      this.props.lines,
      norm,
      interpolate,
      dataX,
      selectedIds,
    );

    GraphModeLinesHover.moveHoverDiv(this._el._hover, this._el._el, chartAreaX, chartAreaOffsetLeft);
  };

  onLeave = ({ chartLines }) => {
    GraphModeLinesHover.killHover(this._el._hover, chartLines);
  };

  onClick = (event) => {
    if (BrowserUtils.inIframe()) {
      graphClickToExit(this.props.url);
      return;
    }

    const { dataX } = event;

    const beginSeconds = Math.trunc(TimeFormatUtils.parseUtc(this.props.graphData.start) / 1000);
    const endSeconds = Math.trunc(TimeFormatUtils.parseUtc(this.props.graphData.end) / 1000);
    let deltaSeconds = endSeconds - beginSeconds;
    if (deltaSeconds < 1800) {
      deltaSeconds *= 2;
    } else if (deltaSeconds < 3600) {
      deltaSeconds = 3600;
    }

    const newBeginSeconds = dataX - (deltaSeconds / 4);
    const newEndSeconds = dataX + (deltaSeconds / 4);

    GraphModeLines.goToLocal(newBeginSeconds, newEndSeconds);
  };

  onSelect = (range) => {
    GraphModeLines.goToLocal(range[0], range[1]);
  };

  render() {
    const {
      graphConf, graphData, lines, width, isLocal,
    } = this.props;

    // Special hack to prevent selection in iframes
    const preventSelection = window.location.pathname.endsWith('/status')
      || window.location.pathname.endsWith('/status/');

    const healthValue = graphData.health.value;

    const conf = GraphConfUtils.toChartConf(graphConf);

    conf.xaxis = XAxisConfUtils.createXaxisConf(graphData, width);

    const borderColor = HealthAlerting.computeBorderColor(healthValue);

    // noinspection UnnecessaryLocalVariableJS
    const hasMinYaxisAreaWidth = isLocal;

    return (
      <ReactChartLines
        lines={lines}
        conf={conf}
        chartBorderWidth={2}
        hasMinYaxisAreaWidth={hasMinYaxisAreaWidth}
        borderColor={borderColor}
        onHover={this.onHover}
        onLeave={this.onLeave}
        onClick={preventSelection ? null : this.onClick}
        onSelect={preventSelection ? null : this.onSelect}
        ref={(el) => { this._el = el; }}
      />
    );
  }
}

GraphModeLines.propTypes = {
  url: PropTypes.string.isRequired,
  lines: PropTypes.array.isRequired,
  graphData: PropTypes.object.isRequired,
  graphConf: PropTypes.object.isRequired,
  graphFormatter: PropTypes.object.isRequired,
  width: PropTypes.number.isRequired,
  graphOnly: PropTypes.bool.isRequired,
  isLocal: PropTypes.bool.isRequired,
};

GraphModeLines.contextTypes = {
  router: PropTypes.shape({
    history: PropTypes.object.isRequired,
  }).isRequired,
};

export default GraphModeLines;
