/* eslint-disable no-param-reassign,operator-assignment */
import ChartCommon from '../common/ChartCommon';
import ChartValueHover from '../hover/ChartValueHover';
import SvgUtils from '../svg/SvgUtils';
import SvgDBuilder from '../common/SvgDBuilder';
import UserLinksBasic from '../../utils/UserLinksBasic';
import TextFit from '../common/TextFit';

class ChartPie {
  /**
   * @param {Element} $rootSvg
   * @param {Element} $rootDiv
   */
  constructor($rootSvg, $rootDiv) {
    this.$rootSvg = $rootSvg;
    this.$rootDiv = $rootDiv;
  }

  static valueToAngle(value, sum) {
    return ChartPie.normalizeAngle(((value / sum) * 2 * Math.PI) - (Math.PI / 2));
  }

  static normalizeAngle(a) {
    while (a < 0) {
      a += 2 * Math.PI;
    }
    a = a % (2 * Math.PI);
    return a;
  }

  /**
   * @param {GraphFormatter} graphFormatter
   * @param {Element} $where
   * @param {AggrValue[]} values
   */
  static chart(
    graphFormatter,
    $where,
    values,
  ) {
    const $rootDiv = ChartCommon.createRootDiv($where);
    const $rootSvg = ChartCommon.createRootSvg($rootDiv);

    const chartPie = new ChartPie($rootSvg, $rootDiv);

    chartPie.drawPies(graphFormatter, $where, values);

    return chartPie;
  }

  drawHover(graphFormatter, value) {
    ChartValueHover.drawHover(value, 2, 0, this.$rootDiv, graphFormatter);
  }

  killHover() {
    ChartValueHover.killHover(this.$rootDiv);
  }

  drawPies(graphFormatter, $where, values) {
    const showValues = ChartCommon.filterNotNanAndNotZero(ChartCommon.filterShow(values));

    const textBox = SvgUtils.measureTextSize('XXX');
    const labelHeight = Math.trunc(textBox.height);

    let sum = 0;
    for (let i = 0; i < showValues.length; ++i) {
      const aggrValue = showValues[i];
      sum += Math.abs(aggrValue.value);
    }

    const width = $where.clientWidth;
    const height = $where.clientHeight;
    const x0 = width / 2;
    const y0 = height / 2;
    const rect = Math.min(width, height);
    const r = rect / 3;
    const textR = r + 20;

    let aggr = 0;

    let lastY = 0;
    let lastLeft = false;
    let isFirstPie = true;

    for (let i = 0; i < showValues.length; ++i) {
      const aggrValue = showValues[i];

      const absValue = Math.abs(aggrValue.value);
      const nextAggr = aggr + absValue;

      const middleAggr = (aggr + nextAggr) / 2;

      const $path = SvgUtils.createPathElement();
      if ((aggr < sum * 0.0001) && (nextAggr > sum * 0.9999)) {
        // Full-circle chunk has begin angle==end angle
        // and thus is not handled correctly by SvgDBuilder::pie
        $path.setAttribute('d', ChartPie.circle(x0, y0, r));
      } else {
        const angle1 = ChartPie.valueToAngle(aggr, sum);
        const angle2 = ChartPie.valueToAngle(nextAggr, sum);
        const d = ChartPie.pie(x0, y0, r, angle1, angle2);
        $path.setAttribute('d', d);
      }

      $path.setAttribute('class', 'chart-pie-pie');
      $path.setAttribute('fill', aggrValue.color);

      const elements = [];

      const $pathA = aggrValue.link
        ? ChartPie.appendA(this.$rootSvg, UserLinksBasic.fixOldAdminUrl(aggrValue.link), '_top')
        : this.$rootSvg;

      $pathA.appendChild($path);

      elements.push($path);

      const percentsNum = (absValue / sum) * 100;
      const percents = `${percentsNum.toFixed(1)}%`;
      aggrValue.percents = percents;

      if (absValue / sum > 0.05) {
        const textAngle = ChartPie.valueToAngle(middleAggr, sum);
        const left = ChartPie.normalizeAngle(textAngle + (Math.PI / 2)) > Math.PI;
        const x = Math.trunc(x0 + (textR * Math.cos(textAngle)));
        const y = Math.trunc(y0 + (textR * Math.sin(textAngle)));
        const maxLabelWidth = left ? x : width - x;

        const $text = SvgUtils.createTextElement();
        $text.setAttribute('x', `${x}`);
        $text.setAttribute('y', `${y}`);
        $text.setAttribute('text-anchor', left ? 'end' : 'begin');
        $text.setAttribute('class', 'chart-pie-label');
        $text.setAttribute('fill', aggrValue.color);

        const labelPrefix = aggrValue.label;
        const percentPostfix = ` ${percents}`;
        const label = TextFit.fitTextWithDotsIfNeeded(
          maxLabelWidth, labelPrefix, percentPostfix, 'chart-pie-label',
        );

        SvgUtils.appendText($text, label);

        const intersectLastPie = !isFirstPie
          && left === lastLeft
          && Math.abs(y - lastY) < labelHeight;

        const inrersectTopOrBottom = y - labelHeight < 0 || y >= height;
        const showPie = !!label && !intersectLastPie && !inrersectTopOrBottom;

        if (showPie) {
          isFirstPie = false;
          lastY = y;
          lastLeft = left;

          const $textA = aggrValue.link
            ? ChartPie.appendA(this.$rootSvg, UserLinksBasic.fixOldAdminUrl(aggrValue.link), '')
            : this.$rootSvg;

          $textA.appendChild($text);

          elements.push($text);
        }
      }

      elements.forEach(($element) => {
        $element.addEventListener('mouseenter', () => this.drawHover(graphFormatter, aggrValue));
        $element.addEventListener('mouseleave', () => this.killHover());
      });

      aggr = nextAggr;
    }
  }

  static appendA($where, href, target) {
    const a = SvgUtils.createElement('a');
    if (href) {
      a.setAttributeNS('http://www.w3.org/1999/xlink', 'href', href);
    }
    if (target) {
      a.setAttribute('target', target);
    }
    $where.appendChild(a);
    return a;
  }

  static pie(x0, y0, r, a1, a2) {
    const a = ChartPie.normalizeAngle(a2 - a1);
    const largeArc = a > Math.PI;

    const d = new SvgDBuilder();
    d.move(x0, y0);
    d.line(x0 + (r * Math.cos(a1)), y0 + (r * Math.sin(a1)));
    d.arc(r, r, 0, largeArc, true, x0 + (r * Math.cos(a2)), y0 + (r * Math.sin(a2)));
    d.loop();

    return d.format();
  }

  static circle(x0, y0, r) {
    const d = new SvgDBuilder();
    d.move(x0 + r, y0);
    // If arc start point equals arc end point svg draws nothing.
    // So to draw a full circle we need two arcs
    d.arc(r, r, 0, false, true, x0 - r, y0);
    d.arc(r, r, 0, true, true, x0 + r, y0);
    d.loop();
    return d.format();
  }
}

export default ChartPie;
