/* eslint-disable import/no-cycle */
import isEqual from 'lodash/isEqual';
import StackBuilder from './StackBuilder';
import TimelineNavigator from './TimelineNavigator';
import Triples from './Triples';
import GraphDataUtils from './GraphDataUtils';

class Baseline {
  /** Millis */
  timestamps;

  // no NaNs
  triples;

  /**
   * @param {LongArrayView} timestamps
   * @param {Triples} triples
   * @constructor
   */
  constructor(timestamps, triples) {
    this.timestamps = timestamps;
    this.triples = triples;
    this.quickValidate();
  }

  /**
   * @param {LongArrayView} timestamps
   * @param {double[]} lefts
   * @param {double[]} middles
   * @param {double[]} rights
   * @return {Baseline}
   */
  static fromTripples(timestamps, lefts, middles, rights) {
    return new Baseline(timestamps, new Triples(lefts, middles, rights));
  }

  /**
   * @description for test only
   * @param {long[]} timestamps
   * @param {double[]} middles
   * @return {Baseline}
   */
  static fromMiddles(timestamps, middles) {
    const newMiddles = [...middles];
    const lefts = [...middles];
    const rights = [...middles];

    if (middles.length > 0) {
      lefts[0] = 0;
      rights[rights.length - 1] = 0;
    }

    const triples = new Triples(lefts, newMiddles, rights);

    return new Baseline(timestamps, triples);
  }

  /**
   * @description must have no NaNs
   * @param {GraphData} graphData
   * @return {Baseline}
   */
  static fromGraphData(graphData) {
    const filteredGraphData = GraphDataUtils.filterNonNan(graphData);
    const { timestamps } = filteredGraphData;
    const middles = filteredGraphData.values;
    return Baseline.fromMiddles(timestamps, middles);
  }

  /**
   * @param {long[]} timestamps
   * @param {double[][]} points
   * @return {Baseline}
   */
  static forTest(timestamps, points) {
    if (timestamps.length !== points.length) {
      throw new Error('different timestamps and points length');
    }

    const lefts = points.map((p) => p[0]);
    const middles = points.map((p) => p[1]);
    const rights = points.map((p) => p[2]);

    const triples = new Triples(lefts, middles, rights);

    return new Baseline([...timestamps], triples);
  }

  /**
   * @return {Baseline}
   */
  static empty() {
    return Baseline.fromTripples(
      [],
      [],
      [],
      [],
    );
  }

  /**
   * @param {DoubleUnaryOperator} op
   * @return {Baseline}
   */
  mapValues(op) {
    return new Baseline(this.timestamps, this.triples.mapValues(op));
  }

  /**
   * @return {boolean}
   */
  isEmpty() {
    return this.timestamps.length === 0;
  }

  quickValidate() {
    if (this.triples.lefts.length !== this.timestamps.length) {
      throw new Error('different size of lefts');
    }

    if (this.timestamps.length > StackBuilder.POINT_LIMIT_SANITY) {
      throw new Error(`Too many points in baseline: ${this.timestamps.length} the limit is ${StackBuilder.POINT_LIMIT_SANITY}`);
    }

    if (this.timestamps.length === 0) {
      return;
    }

    if (this.triples.lefts[0] !== 0) {
      throw new Error('first point of lefts is not 0');
    }
    if (this.triples.rights[this.triples.rights.length - 1] !== 0) {
      throw new Error('last point of rights is not 0');
    }
  }

  getTimestamps() {
    return this.timestamps;
  }

  getTriples() {
    return this.triples;
  }

  getLefts() {
    return this.triples.lefts;
  }

  getMiddles() {
    return this.triples.middles;
  }

  getRights() {
    return this.triples.rights;
  }

  add(ts, left, middle, right) {
    this.timestamps.push(ts);
    this.triples.lefts.push(left);
    this.triples.middles.push(middle);
    this.triples.rights.push(right);
  }

  /**
   * @param {Timeline} timeline
   * @param {Interpolate} interpolate
   * @return {Baseline}
   */
  resample(timeline, interpolate) {
    if (isEqual(this.timestamps, timeline)) {
      return this;
    }

    // optimization
    if (timeline.length === 0) {
      return Baseline.empty();
    }

    const middles = new Array(timeline.length).fill(0);
    const rights = new Array(timeline.length).fill(0);
    const lefts = new Array(timeline.length).fill(0);

    const nav = new TimelineNavigator(this.timestamps);

    for (let i = 0; i < timeline.length; ++i) {
      const tsMillis = timeline[i];
      nav.scrollToMillis(tsMillis);
      const pos = nav.getPos();
      if (nav.isOnPoint()) {
        lefts[i] = this.triples.lefts[pos];
        middles[i] = this.triples.middles[pos];
        rights[i] = this.triples.rights[pos];
      } else if (nav.isBetweenPoints()) {
        const value = interpolate.interpolate(
          this.timestamps[pos], this.triples.rights[pos],
          this.timestamps[pos + 1], this.triples.lefts[pos + 1],
          tsMillis,
        );
        lefts[i] = value;
        middles[i] = value;
        rights[i] = value;
      } else {
        lefts[i] = 0;
        middles[i] = 0;
        rights[i] = 0;
      }
    }

    if (timeline.length > 0) {
      lefts[0] = 0;
      rights[rights.length - 1] = 0;
    }

    return Baseline.fromTripples(timeline, lefts, middles, rights);
  }
}

export default Baseline;
