import Interpolate from './Interpolate';
import TimelineNavigator from './TimelineNavigator';

class GraphDataAsFunction {
  constructor(graphData, interpolate) {
    this.interpolate = interpolate;
    this.nav = new TimelineNavigator(graphData.timestamps);
    this.values = graphData.values;
    this.timestamps = graphData.timestamps;
  }

  /**
   * @param {number} leftTsMillis
   * @param {number} tsMillis
   * @param {number} rightTsMillis
   * @return {number}
   */
  computeAt(leftTsMillis, tsMillis, rightTsMillis) {
    let interpolatedResult;

    this.nav.scrollToMillis(tsMillis);
    if (this.nav.isOnPoint()) {
      interpolatedResult = this.values[this.nav.getPos()];
    } else if (this.nav.isBetweenPoints()) {
      interpolatedResult = this.interpolate.interpolate(
        this.timestamps[this.nav.getPos()], this.values[this.nav.getPos()],
        this.timestamps[this.nav.getPos() + 1], this.values[this.nav.getPos() + 1],
        tsMillis,
      );
    } else {
      interpolatedResult = NaN;
    }

    if (isNaN(interpolatedResult)) {
      // Possibly there are some points, and we should not return NaN
      // because we failed to interpolate for some reason.
      return this.computeAverageOnArea(leftTsMillis, tsMillis, rightTsMillis);
    }

    return interpolatedResult;
  }

  /**
   * @param {number} leftTsMillis
   * @param {number} tsMillis
   * @param {number} rightTsMillis
   */
  computeAverageOnArea(leftTsMillis, tsMillis, rightTsMillis) {
    if (this.interpolate === Interpolate.NONE) {
      return NaN;
    }

    let beginTsMillis;
    let endTsMillis;

    let includeBegin;
    let includeEnd;

    switch (this.interpolate) {
      case Interpolate.LEFT:
        beginTsMillis = leftTsMillis;
        endTsMillis = tsMillis;

        includeBegin = false;
        includeEnd = true;

        break;
      case Interpolate.RIGHT:
        beginTsMillis = tsMillis;
        endTsMillis = rightTsMillis;

        includeBegin = true;
        includeEnd = false;

        break;
      case Interpolate.LINEAR:
        beginTsMillis = tsMillis - ((tsMillis - leftTsMillis) / 2);
        endTsMillis = tsMillis + ((rightTsMillis - tsMillis) / 2);

        includeBegin = false;
        includeEnd = true;

        break;
      default:
        throw new Error(`unknown interpolate: ${this.interpolate}`);
    }

    let sumPointsInArea = 0;
    let countPointsInArea = 0;

    this.nav.scrollToMillis(beginTsMillis);

    while (this.nav.getPos() < this.timestamps.length) {
      const ts = this.timestamps[this.nav.getPos()];

      const isAfterAreaEnd = ts > endTsMillis
        || (ts === endTsMillis && !(includeEnd || (includeBegin && ts === beginTsMillis)));

      if (isAfterAreaEnd) {
        break;
      }

      if (ts > beginTsMillis || (includeBegin && ts === beginTsMillis)) {
        const value = this.values[this.nav.getPos()];
        if (!isNaN(value)) {
          sumPointsInArea += value;
          // eslint-disable-next-line no-plusplus
          countPointsInArea++;
        }
      }

      const oldPos = this.nav.getPos();
      if (oldPos === this.timestamps.length - 1) {
        break;
      }
      this.nav.scrollToMillis(Math.min(endTsMillis, this.timestamps[this.nav.getPos() + 1]));
      if (oldPos === this.nav.getPos()) {
        break;
      }
    }
    if (countPointsInArea !== 0) {
      return sumPointsInArea / countPointsInArea;
    }
    return NaN;
  }
}

export default GraphDataAsFunction;
