import Interpolate from './Interpolate';
import Triples from './Triples';
import StripeIntermediate from './StripeIntermediate';

function nans(size) {
  const r = [];
  for (let i = 0; i < size; ++i) {
    r.push(NaN);
  }
  return r;
}

class MakeStripeOverBaseline {
  /**
   * @param {Baseline} baseline
   * @param {GraphData} graphData
   * @param {Interpolate} interpolate
   * @return {StripeIntermediate}
   */
  static make(baseline, graphData, interpolate) {
    // sanity check
    if (baseline.timestamps.length !== graphData.timestamps.length) {
      throw new Error('failed to create MakeStripeOverBaseline');
    }

    const hl = nans(graphData.values.length);
    const hm = nans(graphData.values.length);
    const hr = nans(graphData.values.length);
    const ll = nans(graphData.values.length);
    const lm = nans(graphData.values.length);
    const lr = nans(graphData.values.length);

    for (let i = 0; i < baseline.timestamps.length; ++i) {
      ll[i] = baseline.getLefts()[i];
      lm[i] = baseline.getMiddles()[i];
      lr[i] = baseline.getRights()[i];

      const value = graphData.values[i];

      if (isNaN(value)) {
        // eslint-disable-next-line no-continue
        continue;
      }

      const isGapAtLeft = i === 0 || isNaN(graphData.values[i - 1]);

      if (!isGapAtLeft && interpolate !== Interpolate.NONE) {
        if (interpolate === Interpolate.LEFT) {
          hl[i] = ll[i] + graphData.values[i - 1];
        } else if (interpolate === Interpolate.LINEAR) {
          hl[i] = ll[i] + value;
        } else if (interpolate === Interpolate.RIGHT) {
          hl[i] = ll[i] + value;
        } else {
          throw new Error('unknown interpolate');
        }
      }

      hm[i] = lm[i] + value;

      // eslint-disable-next-line max-len
      const isGapAtRight = i === graphData.values.length - 1 || isNaN(graphData.values[i + 1]);

      if (!isGapAtRight && interpolate !== Interpolate.NONE) {
        if (interpolate === Interpolate.RIGHT) {
          hr[i] = lr[i] + graphData.values[i + 1];
        } else if (interpolate === Interpolate.LINEAR) {
          hr[i] = lr[i] + value;
        } else if (interpolate === Interpolate.LEFT) {
          hr[i] = lr[i] + value;
        } else {
          throw new Error('unknown interpolate');
        }
      }
    }

    return new StripeIntermediate(
      baseline.timestamps,
      new Triples(hl, hm, hr),
      new Triples(ll, lm, lr),
    );
  }
}

export default MakeStripeOverBaseline;
