/* eslint-disable max-len,no-restricted-syntax,import/no-cycle */
import StackBuildResult from './StackBuildResult';
import Baseline from './Baseline';
import TimelineUtils from './TimelineUtils';
import GraphDataWithStackConfigAndCookie from './GraphDataWithStackConfigAndCookie';
import BaselineMath from './BaselineMath';
import BaselineMax from './BaselineMax';
import GraphDataToBaseline from './GraphDataToBaseline';
import MakeStripeOverBaseline from './MakeStripeFromBaseline';
import Resampler from './Resampler';
import GroupByStack from './GroupByStack';
import GraphDataToStripe from './GraphDataToStripe';
import StackLines from './StackLines';
import GraphDataUtils from './GraphDataUtils';

class StackBuilder {
  static POINT_LIMIT_SANITY = 100000;

  /**
   * @param {List<GraphDataWithStackConfigAndCookie<C>>} inputs
   * @param {Interpolate} interpolate
   * @param {boolean} normalize
   * @return {List<Pair<Stripe, C>>}
   */
  static makeStacks(
    inputs,
    interpolate,
    normalize,
  ) {
    const linesWithOrder = [];
    for (let i = 0; i < inputs.length; ++i) {
      const item = inputs[i];
      linesWithOrder.push(new GraphDataWithStackConfigAndCookie(
        item.graphData,
        item.stack,
        item.down,
        i,
        item.yaxisPositionValue,
      ));
    }

    const linesByStacks = GroupByStack.groupLinesByStack(linesWithOrder);

    let r = [];
    let overallMax = normalize ? Baseline.empty() : null;

    for (const stackLines of linesByStacks) {
      const realignedStackLines = StackBuilder.alignLinesToSameTimeline(stackLines, interpolate);

      let stackBuildResult = StackBuilder.makeStack(realignedStackLines, interpolate);

      if (realignedStackLines.down) {
        stackBuildResult = stackBuildResult.neg();
      }

      const stripes = stackBuildResult.stack;
      if (normalize) {
        overallMax = BaselineMax.maxTwo(overallMax, stackBuildResult.upperBound);
      }

      r.push(...stripes);
    }

    if (normalize) {
      const overallMax1 = overallMax;
      r = r.map((pair) => [
        pair[0].normalize(overallMax1, interpolate),
        pair[1]]);
    }

    const result = r.sort((a, b) => a[1] - b[1])
      .map((l) => {
        const key = l[0];
        const value = l[1];
        const input = inputs[value];
        const stripe = key.toStripe();
        return [stripe, input.cookie];
      });

    return result;
  }

  /**
   * @param {StackLines} input
   * @param {Interpolate} interpolate
   * @return {StackLines}
   */
  static alignLinesToSameTimeline(input, interpolate) {
    // Without this operation NaNs may be aligned close enough to data points thus disappearing during the lines resampling step
    // eslint-disable-next-line no-param-reassign
    input = input.sparseNaNs();

    const commonTimeline = input.makeCommonTimeline();

    const realigned = input.lines
      .map((gd) => new GraphDataWithStackConfigAndCookie(
        Resampler.resample(gd.graphData, commonTimeline, interpolate),
        gd.stack,
        gd.down,
        gd.cookie,
        gd.yaxisPositionValue,
      ));

    return new StackLines(input.stackId, input.down, realigned);
  }

  /**
   * @param {StackLines<C>} lines
   * @param {Interpolate} interpolate
   * @return {StackBuildResult<C>}
   */
  static makeStack(lines, interpolate) {
    const polylines = [];

    for (const line of lines.lines) {
      polylines.push([line.graphData, line.cookie]);
    }

    if (polylines.length === 0) {
      return new StackBuildResult([], Baseline.empty());
    }

    if (polylines.length === 1) {
      const single = polylines[0];
      const stripe = GraphDataToStripe.graphDataToStripe(single[0], interpolate);

      const stack = [[stripe, single[1]]];
      return new StackBuildResult(stack, Baseline.fromGraphData(GraphDataUtils.filterNonNan(single[0])));
    }

    const r = [];

    const commonTimeline = lines.makeCommonTimeline();
    const graphData = {
      timestamps: commonTimeline,
      values: new Array(commonTimeline.length).fill(0),
    };
    let baseline = Baseline.fromGraphData(graphData);

    for (const polyline of polylines) {
      const combine = StackBuilder.combineMatchingTimelines(baseline, polyline[0], interpolate);
      baseline = combine[0];
      r.push([combine[1], polyline[1]]);
    }

    return new StackBuildResult(r, baseline);
  }

  /**
   * @param {Baseline} baseline
   * @param {GraphData} graphData
   * @param {Interpolate} interpolate
   * @return {Pair<Baseline, StripeIntqermediate>}
   */
  static combineMatchingTimelines(
    baseline,
    graphData,
    interpolate,
  ) {
    const graphDataToBaseline1 = GraphDataToBaseline.graphDataToBaseline(graphData, interpolate);
    const nextBaseline = BaselineMath.sum(baseline, graphDataToBaseline1);
    return [nextBaseline, MakeStripeOverBaseline.make(baseline, graphData, interpolate)];
  }

  /**
   * @param {Baseline} baseline
   * @param {GraphData} graphData
   * @param {Interpolate} interpolate
   * @return {Pair<Baseline, StripeIntermediate>}
   */
  static combine(
    baseline,
    graphData,
    interpolate,
  ) {
    const commonTimeline = TimelineUtils.unionTwo(baseline.timestamps, graphData.timestamps);
    const baselineResampled = baseline.resample(commonTimeline, interpolate);
    const graphDataToBaseline1 = GraphDataToBaseline.graphDataToBaseline(graphData, interpolate);
    const graphDataToBaseline = graphDataToBaseline1.resample(commonTimeline, interpolate);
    const graphDataResampled = Resampler.resample(graphData, commonTimeline, interpolate);
    const nextBaseline = BaselineMath.sum(baselineResampled, graphDataToBaseline);
    const stripeIntermediate = MakeStripeOverBaseline.make(baselineResampled, graphDataResampled, interpolate);
    return [nextBaseline, stripeIntermediate];
  }
}

export default StackBuilder;
