import GraphDataAsFunction from './GraphDataAsFunction';

class Stripe {
  /**
   * Millis. May contain repeated values.
   */
  timestamps;

  /**
   * NaN mean gap.
   */
  highs;

  lows;

  /**
   * @param {number[]} timestamps
   * @param {number[]} highs
   * @param {number[]} lows
   * @constructor
   */
  constructor(timestamps, highs, lows) {
    this.timestamps = timestamps;
    this.highs = highs;
    this.lows = lows;
  }

  static fromPoints(points) {
    return new Stripe(
      points.map((p) => p.ts),
      points.map((p) => p.high),
      points.map((p) => p.low),
    );
  }

  getTimestamps() {
    return this.timestamps;
  }

  getHighs() {
    return this.highs;
  }

  getLows() {
    return this.lows;
  }

  /**
   * @param {GraphData} hundreds
   * @param {Interpolate} interpolate
   * @return {Stripe}
   */
  normalize(hundreds, interpolate) {
    const graphDataAsFunction = new GraphDataAsFunction(hundreds, interpolate);

    const newHighs = new Array(this.timestamps.length).fill(0);
    const newLows = new Array(this.timestamps.length).fill(0);
    for (let i = 0; i < this.timestamps.length; ++i) {
      const leftTs = this.timestamps[i === 0 ? i : i - 1];
      const ts = this.timestamps[i];
      const rightTs = this.timestamps[i === (this.timestamps.length - 1) ? i : i + 1];
      const h = graphDataAsFunction.computeAt(leftTs, ts, rightTs);
      if (isNaN(this.highs[i])) {
        newHighs[i] = NaN;
        newLows[i] = NaN;
      } else if (h !== 0) {
        newHighs[i] = (this.highs[i] / h) * 100;
        newLows[i] = (this.lows[i] / h) * 100;
      } else {
        newHighs[i] = 0;
        newLows[i] = 0;
      }
    }

    return new Stripe(this.timestamps, newHighs, newLows);
  }

  /**
   * @param {number} index
   * @return {PointOrStrip}
   */
  getPointHighLow(index) {
    return [
      Math.trunc(this.timestamps[index] / 1000),
      this.highs[index],
      this.lows[index],
    ];
  }

  /**
   * @param {number} index
   * @return {PointOrStrip}
   */
  getPointHigh(index) {
    return [
      Math.trunc(this.timestamps[index] / 1000),
      this.highs[index],
    ];
  }

  /**
   * @return {PointOrStrip[]}
   */
  getPointsHighLow() {
    const result = new Array(this.timestamps.length);
    for (let i = 0; i < this.timestamps.length; ++i) {
      result[i] = this.getPointHighLow(i);
    }
    return result;
  }

  /**
   * @return {PointOrStrip[]}
   */
  getPointsHigh() {
    const result = new Array(this.timestamps.length);
    for (let i = 0; i < this.timestamps.length; ++i) {
      result[i] = this.getPointHigh(i);
    }
    return result;
  }
}

export default Stripe;
