import Triples from './Triples';
import BaselineAsFunction from './BaselineAsFunction';
import StripeBuilder from './StripeBuilder';

class StripeIntermediate {
  /**
   * @param {LongArrayView} timestamps
   * @param {Triples} highs
   * @param {Triples} lows
   * @constructor
   */
  constructor(timestamps, highs, lows) {
    this.timestamps = timestamps;
    this.highs = highs;
    this.lows = lows;
  }

  /**
   * @param {TripleWhich} which
   * @param {number} i
   * @return {boolean}
   */
  has(which, i) {
    return !isNaN(this.getHigh(which, i));
  }

  /**
   * @param {TripleWhich} which
   * @param {number} i
   * @return {number}
   */
  getHigh(which, i) {
    return this.highs[which][i];
  }

  /**
   * @param {TripleWhich} which
   * @param {number} i
   * @return {number}
   */
  getLow(which, i) {
    return this.lows[which][i];
  }

  mapValues(f) {
    return new StripeIntermediate(this.timestamps, this.highs.mapValues(f), this.lows.mapValues(f));
  }

  toStripe() {
    const r = new StripeBuilder();

    for (let i = 0; i < this.timestamps.length; ++i) {
      // eslint-disable-next-line no-restricted-syntax
      for (const which of ['lefts', 'middles', 'rights']) {
        const ts = this.timestamps[i];
        if (!this.has(which, i)) {
          r.enqueueGap(ts);
        } else {
          r.addRealPoint(ts, this.getHigh(which, i), this.getLow(which, i));
        }
      }
    }

    return r.build();
  }

  /**
   * @param {Baseline} hundreds
   * @param {Interpolate} interpolate
   * @return {StripeIntermediate}
   */
  normalize(hundreds, interpolate) {
    const graphDataAsFunction = new BaselineAsFunction(hundreds, interpolate);

    const newHighs = Triples.forSize(this.timestamps.length);
    const newLows = Triples.forSize(this.timestamps.length);
    for (let i = 0; i < this.timestamps.length; ++i) {
      const ts = this.timestamps[i];

      // eslint-disable-next-line no-restricted-syntax
      for (const which of ['lefts', 'middles', 'rights']) {
        const h = graphDataAsFunction.computeAt(ts, which);
        if (!this.has(which, i)) {
          newHighs[which][i] = NaN;
          newLows[which][i] = NaN;
        } else if (h !== 0) {
          newHighs[which][i] = (this.highs[which][i] / h) * 100;
          newLows[which][i] = (this.lows[which][i] / h) * 100;
        } else {
          newHighs[which][i] = 0;
          newLows[which][i] = 0;
        }
      }
    }

    return new StripeIntermediate(this.timestamps, newHighs, newLows);
  }
}

export default StripeIntermediate;
