class SvgUtils {
  /**
   * Create svg element
   *
   * @param name tag name ("svg", "path", "rect", etc.)
   * @return {Element} new svg element
   */
  static createElement(name) {
    return document.createElementNS('http://www.w3.org/2000/svg', name);
  }

  /**
   * Create <svg> element
   * @return {SVGElement} new <svg> element
   */
  static createSvgElement() {
    return SvgUtils.createElement('svg');
  }

  /**
   * Create <Rect> element
   * @return {SVGRectElement} new <rect> element
   */
  static createRectElement() {
    return SvgUtils.createElement('rect');
  }

  /**
   * Create <Path> element
   * @return {SVGPathElement} new <path> element
   */
  static createPathElement() {
    return SvgUtils.createElement('path');
  }

  /**
   * Create <Circle> element
   * @return {SVGCircleElement} new <circle> element
   */
  static createCircleElement() {
    return SvgUtils.createElement('circle');
  }

  /**
   * Create <text> element
   * @return {SVGTextElement} new <text> element
   */
  static createTextElement() {
    return SvgUtils.createElement('text');
  }

  /**
   * Append text node to element
   * @param {Element} element
   * @param {string} text
   */
  static appendText($element, text) {
    const $text = document.createTextNode(text);
    $element.appendChild($text);
  }

  /**
   * @param className
   * @param d
   */
  static createPathElementWithD(className, d) {
    const $element = SvgUtils.createPathElement();
    $element.setAttribute('class', className);
    $element.setAttribute('d', d.format());
    return $element;
  }

  /**
   * Measure text size for SVG
   *
   * @param {String} text
   * @param {String} cssClassName
   * @return {{ width: number, height: number }} measured text size
   */
  static measureTextSize(text, cssClassName = '') {
    const svgElement = SvgUtils.createElement('svg');
    document.body.appendChild(svgElement);

    const textElement = SvgUtils.createElement('text');
    if (cssClassName) {
      textElement.setAttribute('class', cssClassName);
    }
    svgElement.appendChild(textElement);
    textElement.appendChild(document.createTextNode(text));

    const r = SvgUtils.measureWidthHeightOrDefault(textElement);

    svgElement.parentElement.removeChild(svgElement);

    return r;
  }

  /*
      getBBox() throws NS_ERROR_FAILURE exception in Firefox if element doesn't render
      (see https://bugzilla.mozilla.org/show_bug.cgi?id=612118).
      So it's necessary to show default (0x0) widthHeight as Chromium does.

  */

  /**
   * Measure text element
   *
   * @param {SVGTextElement} textElement
   * @return {{ width: number, height: number }} width and height size element
   */
  static measureWidthHeightOrDefault(textElement) {
    try {
      return textElement.getBBox();
    } catch (ignored) {
      console.info('Failed to measure text, so we return 0x0 size');
      return { width: 0, height: 0 };
    }
  }
}

export default SvgUtils;
