import * as TimeUnit from './TimeUnit';
import Weeks from './Weeks';

class TicksMaker {
  static roundValueDown(value, stepUnit) {
    const min = stepUnit.getType() === TimeUnit.DAY_MILLIS ? 1 : 0;
    // eslint-disable-next-line max-len
    return Math.trunc((Math.trunc((value - min) / stepUnit.getMultiplier()) * stepUnit.getMultiplier()) + min);
  }

  static roundDown(date, stepUnit) {
    switch (stepUnit.getType()) {
      case TimeUnit.YEAR_MILLIS:
        return new Date(TicksMaker.roundValueDown(date.getFullYear(), stepUnit), 0);
      case TimeUnit.MONTH_MILLIS:
        return new Date(date.getFullYear(), TicksMaker.roundValueDown(date.getMonth(), stepUnit));
      case TimeUnit.WEEK_MILLIS:
        return Weeks.roundToWeekStart(date, stepUnit.getMultiplier());
      case TimeUnit.DAY_MILLIS: {
        const day = TicksMaker.roundValueDown(date.getDate(), stepUnit);
        return new Date(date.getFullYear(), date.getMonth(), day);
      }
      case TimeUnit.HOUR_MILLIS: {
        const hour = TicksMaker.roundValueDown(date.getHours(), stepUnit);
        return new Date(date.getFullYear(), date.getMonth(), date.getDate(), hour);
      }
      case TimeUnit.MINUTE_MILLIS: {
        const minutes = TicksMaker.roundValueDown(date.getMinutes(), stepUnit);
        return new Date(
          date.getFullYear(),
          date.getMonth(),
          date.getDate(),
          date.getHours(),
          minutes,
        );
      }
      default:
        throw new Error(`unknown range unit: ${stepUnit.getType()}`);
    }
  }

  static tickMarks(interval, stepUnit) {
    const r = [];
    let dt = TicksMaker.roundDown(interval.begin, stepUnit);
    while (dt.getTime() <= interval.end.getTime()) {
      if (dt.getTime() >= interval.begin.getTime()) {
        r.push(dt);
      }

      if (stepUnit.getType() === TimeUnit.MONTH_MILLIS) {
        dt = new Date(dt.getTime());
        dt.setMonth(dt.getMonth() + stepUnit.getMultiplier());
      } else if (stepUnit.getType() === TimeUnit.YEAR_MILLIS) {
        dt = new Date(dt.getTime());
        dt.setFullYear(dt.getFullYear() + stepUnit.getMultiplier());
      } else {
        dt = new Date(dt.getTime() + (stepUnit.getMultiplier() * stepUnit.getType()));
      }
    }
    return r;
  }
}

export default TicksMaker;
