import { parseISO8601 } from '../../utils/iso8601';
import Weekends from './Weekends';
import * as TimeUnit from './TimeUnit';
import StepUnit from './StepUnit';
import Xaxis from './Xaxis';

const MAX_TICK_LABEL_WIDTH = 65;

function makeWeekends(interval, stepUnit) {
  let weekends;

  if (stepUnit.getType() <= TimeUnit.WEEK_MILLIS) {
    const weekendIntervals = Weekends.weekendsIntersectingInterval(interval);
    weekends = weekendIntervals.map((weekend) => ({
      begin: weekend.begin.getTime() / 1000,
      end: weekend.end.getTime() / 1000,
    }));
  } else {
    weekends = [];
  }

  return weekends;
}

export function createXaxisConf(graphData, clientWidth) {
  const xaxisConf = {};

  const beginMillis = parseISO8601(graphData.start);
  const endMillis = parseISO8601(graphData.end);
  const nowMillis = parseISO8601(graphData.now);

  xaxisConf.min = beginMillis / 1000;
  xaxisConf.max = endMillis / 1000;
  xaxisConf.now = nowMillis / 1000;

  const maxXTicks = clientWidth / MAX_TICK_LABEL_WIDTH;

  const interval = {
    begin: new Date(beginMillis),
    end: new Date(endMillis),
  };

  const stepUnit = StepUnit.forInterval(interval, maxXTicks);

  const weekends = makeWeekends(interval, stepUnit);

  const ticks = Xaxis.makeTicks(interval, stepUnit);

  xaxisConf.weekends = weekends;
  xaxisConf.ticks = ticks;

  return xaxisConf;
}

export default { createXaxisConf };
