import omit from 'lodash/omit';

import * as API from '../../../api/alerts';
import { copyFieldsFromTemplate, ONLY_TEMPLATE_FIELDS } from '../../../pages/alerts/constants';
import { addDangerAlert } from '../toaster';

export const LOAD_ALERT = 'LOAD_ALERT';
const SAVE_ALERT = 'SAVE_ALERT';
const SAVE_ALERTS_FROM_TEMPLATES = 'SAVE_ALERTS_FROM_TEMPLATES';
const CHANGE_ALERT_STATE = 'CHANGE_ALERT_STATE';
const DELETE_ALERT = 'DELETE_ALERT';
const CLEAR_ALERT = 'CLEAR_ALERT';

export const SERVICE_FIELDS = ['templateData', 'realType', 'realAlert'];

// eslint-disable-next-line max-len
export const loadAlert = (projectId, alertId) => (dispatch) => API.fetchAlert(projectId, alertId).then(
  async (resp) => {
    if (resp.type.fromTemplate) {
      const template = resp.type.fromTemplate;

      return API.fetchAlertTemplate(
        template.templateId,
        template.templateVersionTag,
      ).then((tResp) => {
        const alert = copyFieldsFromTemplate({
          ...resp,
          realType: resp.type,
          templateData: tResp,
        });

        return dispatch({
          type: LOAD_ALERT,
          payload: alert,
        });
      });
    }

    return dispatch({ type: LOAD_ALERT, payload: resp });
  },
  (resp) => dispatch(addDangerAlert(resp.message)),
);

export const loadAlertInterpolated = (projectId, alertId) => (dispatch) =>
  API.fetchAlert(projectId, alertId)
    .then(async (resp) => {
      const realType = resp.type;
      if (resp.type.fromTemplate) {
        const template = resp.type.fromTemplate;

        return Promise.all([
          API.fetchAlertTemplate(
            template.templateId,
            template.templateVersionTag,
          ),
          API.fetchAlertInterpolated(projectId, alertId),
        ]).then(([templateData, { interpolatedAlert = {} }]) => dispatch({
          type: LOAD_ALERT,
          payload: {
            ...interpolatedAlert,
            templateData,
            fromTemplate: true,
            realType,
            realAlert: resp,
          },
        }));
      }

      return dispatch({ type: LOAD_ALERT, payload: resp });
    },
    (resp) => dispatch(addDangerAlert(resp.message)));

export const loadAlertTemplate = (templateId, versionTagId) => (dispatch) => API.fetchAlertTemplate(
  templateId,
  versionTagId,
).then(
  (resp) => dispatch(({ type: LOAD_ALERT, payload: { templateData: resp } })),
  (resp) => dispatch(addDangerAlert(resp.message)),
);

export const loadAlertTemplateLastVersion = (id) => (dispatch) => API.fetchAlertTemplateLast(
  id,
).then(
  (resp) => dispatch(({ type: LOAD_ALERT, payload: { templateData: resp } })),
  (resp) => dispatch(addDangerAlert(resp.message)),
);

export const setAlert = (alert) => (dispatch) => dispatch({ type: LOAD_ALERT, payload: alert });

export const saveAlert = (projectId, alert, isNew) => (dispatch) => {
  const promise = isNew ? API.createAlert(projectId, alert) : API.updateAlert(projectId, alert);
  promise.then(
    (resp) => dispatch({ type: SAVE_ALERT, payload: resp }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );
  return promise;
};

export const saveAlerts = (projectId, payload) => (dispatch) => {
  const promise = API.createTemplateAlerts(projectId, payload);
  promise.then(
    (resp) => dispatch({ type: SAVE_ALERTS_FROM_TEMPLATES, payload: resp }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );
  return promise;
};

export const changeAlertState = (projectId, alert, newState) => (dispatch) => {
  const realAlert = alert.realAlert || alert;
  const newAlert = { ...realAlert, state: newState };
  delete newAlert.notificationChannels;

  if (newAlert.type.fromTemplate) {
    ONLY_TEMPLATE_FIELDS.forEach((field) => {
      const fieldName = Array.isArray(field) ? field[0] : field;
      delete newAlert[fieldName];
    });
  }

  delete newAlert.fromTemplate;

  return API.updateAlert(projectId, omit(newAlert, SERVICE_FIELDS))
    .then(
      (resp) => dispatch({ type: CHANGE_ALERT_STATE, payload: resp }),
      (resp) => dispatch(addDangerAlert(resp.message)),
    );
};

export const deleteAlert = (projectId, alertId) => (dispatch) => API.deleteAlert(projectId, alertId)
  .then(
    (resp) => dispatch({ type: DELETE_ALERT, payload: resp }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );

export const clearAlert = () => (dispatch) => dispatch({ type: CLEAR_ALERT });

export const alertReducer = (state = {}, action) => {
  switch (action.type) {
    case LOAD_ALERT:
    case CHANGE_ALERT_STATE:
      if (state.templateData) {
        return { ...action.payload, templateData: state.templateData };
      }

      return action.payload;
    case SAVE_ALERT:
    case SAVE_ALERTS_FROM_TEMPLATES:
    case CLEAR_ALERT:
    case DELETE_ALERT:
      return {};
    default:
      return state;
  }
};
