import * as API from '../../../api/alerts';

const START_TO_LOAD_ALERT_STATES = 'START_TO_LOAD_ALERT_STATES';
const FINISH_TO_LOAD_ALERT_STATES = 'FINISH_TO_LOAD_ALERT_STATES';
const FAILED_TO_LOAD_ALERT_STATES = 'FAILED_TO_LOAD_ALERT_STATES';
const CLEAR_ALERT_STATES = 'CLEAR_ALERT_STATES';

const loadAlertState = (projectId, alertId, filterByEvalStatus) =>
  API.fetchAlert(projectId, alertId)
    .then((alertResp) => {
      if (alertResp.groupByLabels && alertResp.groupByLabels.length > 0) {
        const params = {
          pageSize: 100,
          annotationKeys: 'trafficLight.value,trafficLight.color',
          filterByEvaluationStatus: filterByEvalStatus.join(','),
        };

        return API.fetchSubAlerts(projectId, alertId, params)
          .then((subAlertsResp) => ({
            alert: alertResp,
            subAlerts: subAlertsResp.items,
          }));
      }
      return API.fetchAlertEvalState(projectId, alertId)
        .then((evalStateResp) => {
          const status = evalStateResp.status.code;
          if (filterByEvalStatus.length > 0 && filterByEvalStatus.indexOf(status) < 0) {
            return null;
          }

          return {
            alert: alertResp,
            evalState: evalStateResp,
          };
        });
    });

export const loadAlertStates = (projectId, alertIds, filterByEvalStatus) => (dispatch) => {
  dispatch({ type: START_TO_LOAD_ALERT_STATES });

  const promises = alertIds.map((alertId) =>
    loadAlertState(projectId, alertId, filterByEvalStatus));

  return Promise.all(promises)
    .then((resp) => dispatch({
      type: FINISH_TO_LOAD_ALERT_STATES,
      payload: resp.filter((alertWithState) => alertWithState !== null),
    }),
    (resp) => dispatch({ type: FAILED_TO_LOAD_ALERT_STATES, payload: resp }));
};

export const clearAlertStates = () => (dispatch) => dispatch({ type: CLEAR_ALERT_STATES });

const INITIAL_ALERT_STATES = {
  loading: false,
  data: null,
  error: null,
};

export const alertStatesReducer = (state = INITIAL_ALERT_STATES, action) => {
  switch (action.type) {
    case START_TO_LOAD_ALERT_STATES:
      return {
        loading: true,
        data: state.data,
        error: null,
      };
    case FINISH_TO_LOAD_ALERT_STATES:
      return {
        loading: false,
        data: action.payload,
        error: null,
      };
    case FAILED_TO_LOAD_ALERT_STATES:
      return {
        loading: false,
        data: null,
        error: action.payload,
      };
    case CLEAR_ALERT_STATES:
      return INITIAL_ALERT_STATES;
    default:
      return state;
  }
};
