import uniqid from 'uniqid';
import * as InternalAPI from '../../../api/internal';
import * as ExperimentalDashboardAPI from '../../../api/experimentalApi/experimentalDashboardApi';
import { addDangerAlert } from '../toaster';

const START_TO_LOAD_DASHBOARD = 'old/dashboard/START_TO_LOAD';
const FINISH_TO_LOAD_DASHBOARD = 'old/dashboard/FINISH_TO_LOAD';
const FAILED_TO_LOAD_DASHBOARD = 'old/dashboard/FAILED_TO_LOAD';
const PIN_DASHBOARD = 'old/dashboard/PIN';
const UNPIN_DASHBOARD = 'old/dashboard/UNPIN';
const CLEAR_DASHBOARD = 'old/dashboard/CLEAR';

function fixDashboard(genericResp) {
  if (!genericResp.dashboard) {
    return genericResp;
  }

  const { dashboard } = genericResp;

  dashboard.rows.forEach((row) => {
    row.items.forEach((item) => {
      // eslint-disable-next-line no-param-reassign
      item.uuid = uniqid('panel_');
    });
  });

  return genericResp;
}

const loadDashboardExperimentally = (dashboardId, params) => (dispatch) => {
  dispatch({ type: START_TO_LOAD_DASHBOARD, payload: { dashboardId } });

  return ExperimentalDashboardAPI.loadDashboard(params).then(
    (resp) => dispatch({
      type: FINISH_TO_LOAD_DASHBOARD,
      payload: { dashboardId, content: fixDashboard(resp) },
    }),
    (resp) => dispatch({
      type: FAILED_TO_LOAD_DASHBOARD,
      payload: { dashboardId, content: resp },
    }),
  );
};

export const loadDashboard = (dashboardId, params) => (dispatch) =>
  dispatch(loadDashboardExperimentally(dashboardId, params));

export const pinDashboard = (dashboardId, params) => (dispatch) =>
  InternalAPI.createPin(params)
    .then(
      (resp) => dispatch({
        type: PIN_DASHBOARD,
        payload: { dashboardId, content: resp.id },
      }),
      (resp) => dispatch(addDangerAlert(resp.message)),
    );

export const unpinDashboard = (dashboardId, pinId) => (dispatch) =>
  InternalAPI.deletePin(pinId)
    .then(
      () => dispatch({ type: UNPIN_DASHBOARD, payload: { dashboardId } }),
      (resp) => dispatch(addDangerAlert(resp.message)),
    );

export const clearDashboard = (dashboardId) => (dispatch) =>
  dispatch({ type: CLEAR_DASHBOARD, payload: { dashboardId } });

const DEFAULT_OLD_DASHBOARD_STATE = {
  loading: false,
  data: null,
  error: null,
};

const oldDashboards = (state = DEFAULT_OLD_DASHBOARD_STATE, action) => {
  switch (action.type) {
    case START_TO_LOAD_DASHBOARD:
      return { ...state, loading: true, error: null };
    case FINISH_TO_LOAD_DASHBOARD:
      return { loading: false, data: action.payload, error: null };
    case FAILED_TO_LOAD_DASHBOARD:
      return { loading: false, data: null, error: action.payload };
    case PIN_DASHBOARD:
      return {
        ...state,
        data: {
          ...state.data,
          currentPageIdInHistory: action.payload,
        },
      };
    case UNPIN_DASHBOARD:
      return {
        ...state,
        data: {
          ...state.data,
          currentPageIdInHistory: null,
        },
      };
    case CLEAR_DASHBOARD:
      return DEFAULT_OLD_DASHBOARD_STATE;
    default:
      return state;
  }
};

export const oldDashboardsReducer = (state = {}, action) => {
  if (!action.payload || !action.payload.dashboardId) {
    return state;
  }

  const { dashboardId, content } = action.payload;

  const oldDashboardState = state[dashboardId] || DEFAULT_OLD_DASHBOARD_STATE;
  const oldDashboardAction = { type: action.type, payload: content };

  return { ...state, [dashboardId]: oldDashboards(oldDashboardState, oldDashboardAction) };
};
