/* eslint-disable no-param-reassign */
import { getColorAlexPetrov, getGradientColorByZone } from '../../../utils/color/colors';
import GradientZone from '../../../utils/color/GradientZone';
import { addDangerAlert } from '../toaster';

import * as InternalAPI from '../../../api/internal';
import * as ExperimentalDataAPI from '../../../api/experimentalApi/experimentalDataApi';

export const START_TO_LOAD_GENERIC = 'old/graph/START_TO_LOAD_GENERIC';
export const FINISH_TO_LOAD_GENERIC = 'old/graph/FINISH_TO_LOAD_GENERIC';
export const FAILED_TO_LOAD_GENERIC = 'old/graph/FAILED_TO_LOAD_GENERIC';
export const FINISH_TO_RELOAD_GRAPH = 'old/graph/FINISH_TO_RELOAD_GRAPH';
export const FAILED_TO_RELOAD_GRAPH = 'old/graph/FAILED_TO_RELOAD_GRAPH';
export const FINISH_TO_LOAD_GRAPH_DATA = 'old/graph/FINISH_TO_LOAD_GRAPH_DATA';
export const FAILED_TO_LOAD_GRAPH_DATA = 'old/graph/FAILED_TO_LOAD_GRAPH_DATA';
const PIN_GRAPH = 'old/graph/PIN_GRAPH';
const UNPIN_GRAPH = 'old/graph/UNPIN_GRAPH';
const CLEAR_GRAPH = 'old/graph/CLEAR';

/* Action creators */

const loadGraphExperimentally = (graphId, params) => async (dispatch) => {
  dispatch({ type: START_TO_LOAD_GENERIC, payload: { graphId } });

  let genericResponse;

  try {
    genericResponse = await ExperimentalDataAPI.loadGeneric(params);
  } catch (e) {
    console.error('failed to load generic data', e);
    dispatch({
      type: FAILED_TO_LOAD_GENERIC,
      payload: { graphId, content: e },
    });
    return;
  }

  dispatch({
    type: FINISH_TO_LOAD_GENERIC,
    payload: { graphId, content: genericResponse },
  });

  const { fullItem } = genericResponse;

  const { timeRange } = genericResponse;

  const fromMillis = timeRange.begin.epochMillis;
  const toMillis = timeRange.end.epochMillis;
  const nowMillis = timeRange.now.epochMillis;

  if (fullItem) {
    let graphDataResponse;

    try {
      graphDataResponse = await ExperimentalDataAPI.loadGraphData(
        params, fullItem, fromMillis, toMillis, nowMillis,
      );
    } catch (e) {
      console.error('failed to load graph data', e);

      dispatch({
        type: FAILED_TO_LOAD_GRAPH_DATA,
        payload: { graphId, content: e },
      });
      return;
    }

    dispatch({
      type: FINISH_TO_LOAD_GRAPH_DATA,
      payload: { graphId, content: graphDataResponse },
    });
  }
};

export const loadGraph = (graphId, params) => (dispatch) =>
  dispatch(loadGraphExperimentally(graphId, params));

const reloadGraphExperimentally = (graphId, params) => async (dispatch) => {
  let genericResponse;

  try {
    genericResponse = await ExperimentalDataAPI.loadGeneric(params, true);
  } catch (e) {
    console.error('failed to reload generic data', e);
    dispatch({
      type: FAILED_TO_RELOAD_GRAPH,
      payload: { graphId, content: e },
    });
    return;
  }

  const { fullItem } = genericResponse;

  const { timeRange } = genericResponse;

  const fromMillis = timeRange.begin.epochMillis;
  const toMillis = timeRange.end.epochMillis;
  const nowMillis = timeRange.now.epochMillis;

  if (fullItem) {
    let graphDataResponse;

    try {
      graphDataResponse = await ExperimentalDataAPI.loadGraphData(
        params, fullItem, fromMillis, toMillis, nowMillis,
      );
    } catch (e) {
      console.error('failed to reload graph data', e);
      dispatch({
        type: FAILED_TO_RELOAD_GRAPH,
        payload: { graphId, content: e },
      });
      return;
    }

    dispatch({
      type: FINISH_TO_RELOAD_GRAPH,
      payload: {
        graphId,
        content: {
          metaData: fullItem,
          graphData: graphDataResponse,
        },
      },
    });
  }
};

export const reloadGraph = (graphId, params) => (dispatch) =>
  dispatch(reloadGraphExperimentally(graphId, params));

export const pinGraph = (graphId, params) => (dispatch) => InternalAPI.createPin(params)
  .then(
    (resp) => dispatch({
      type: PIN_GRAPH,
      payload: { graphId, content: resp.id },
    }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );

export const unpinGraph = (graphId, pinId) => (dispatch) => InternalAPI.deletePin(pinId)
  .then(
    () => dispatch({ type: UNPIN_GRAPH, payload: { graphId } }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );

export const clearGraph = (graphId) => (dispatch) =>
  dispatch({ type: CLEAR_GRAPH, payload: { graphId } });

/* Reducer */

function applyMetadata(graphData, fullItem) {
  const knownMetadataBySalmonId = {};

  const metaDataLines = fullItem.graphData;

  for (let i = 0; i < metaDataLines.length; i++) {
    const line = metaDataLines[i];
    knownMetadataBySalmonId[line.salmonId] = line;
  }

  const graphDataLines = graphData.graphData;

  for (let i = 0; i < graphDataLines.length; i++) {
    const points = graphDataLines[i];
    let meta = knownMetadataBySalmonId[points.salmonId];
    // It's new line that absent in metadata as a result TOP expression
    if (!meta) {
      meta = {
        labels: {},
        area: false,
        yaxisConf: { positionValue: 'left' },
        color: getColorAlexPetrov(i),
      };
    } else {
      points.color = meta.color;
    }

    graphDataLines[i] = { ...meta, ...points };
  }
}

function assignColorsToLines(graphData) {
  for (let i = 0; i < graphData.length; ++i) {
    const series = graphData[i];
    if (series.extendedColor) {
      if (series.extendedColor.fixedColor) {
        series.color = series.extendedColor.color;
      }
      if (series.extendedColor.autoColor) {
        series.color = getColorAlexPetrov(i);
      }
      if (series.extendedColor.gradientColor) {
        let zone;
        switch (series.extendedColor.zone) {
          case 'greenYellow':
            zone = GradientZone.GREEN_YELLOW_VALUE;
            break;
          case 'yellowRed':
            zone = GradientZone.YELLOW_RED_VALUE;
            break;
          case 'redViolet':
            zone = GradientZone.RED_VIOLET_VALUE;
            break;
          default:
            throw new Error(`unknown zone: ${series.extendedColor.zone}`);
        }
        series.color = getGradientColorByZone(series.extendedColor.gradientPosition, zone);
      }
    } else {
      series.color = getColorAlexPetrov(i);
    }
  }
}

const DEFAULT_OLD_GRAPH_GENERIC_STATE = {
  loading: false,
  data: null,
  error: null,
};

const DEFAULT_OLD_GRAPH_DATA_STATE = {
  loading: false,
  data: null,
  error: null,
};

const DEFAULT_OLD_GRAPH_STATE = {
  generic: DEFAULT_OLD_GRAPH_GENERIC_STATE,
  graphData: DEFAULT_OLD_GRAPH_DATA_STATE,
  error: null,
};

const oldGraphs = (state = DEFAULT_OLD_GRAPH_STATE, action) => {
  switch (action.type) {
    case START_TO_LOAD_GENERIC:
      return {
        generic: { ...state.generic, loading: true },
        graphData: state.graphData,
        error: null,
      };
    case FINISH_TO_LOAD_GENERIC: {
      const generic = action.payload;

      if (generic.fullItem && generic.fullItem.graphData) {
        assignColorsToLines(generic.fullItem.graphData);
      }

      return {
        ...state,
        generic: { loading: false, data: generic, error: null },
        graphData: { ...state.graphData, loading: true, error: null },
        error: null,
      };
    }
    case FAILED_TO_LOAD_GENERIC:
      return {
        ...state,
        generic: { loading: false, data: null, error: action.payload },
        error: null,
      };
    case FINISH_TO_LOAD_GRAPH_DATA: {
      const graphData = action.payload;

      const genericData = state.generic.data;

      if (!genericData) {
        return state;
      }

      return {
        ...state,
        graphData: { loading: false, data: graphData, error: null },
        error: null,
      };
    }
    case FAILED_TO_LOAD_GRAPH_DATA:
      return {
        ...state,
        graphData: { ...state.graphData, loading: false, error: action.payload },
        error: null,
      };
    case FINISH_TO_RELOAD_GRAPH: {
      const { graphData, metaData } = action.payload;

      if (metaData && metaData.graphData) {
        assignColorsToLines(metaData.graphData);
        applyMetadata(graphData, metaData);
      }

      return {
        ...state,
        generic: {
          loading: false,
          data: { ...state.generic.data, fullItem: metaData },
          error: null,
        },
        graphData: { loading: false, data: graphData, error: null },
        error: null,
      };
    }
    case FAILED_TO_RELOAD_GRAPH:
      return {
        ...state,
        error: action.payload,
      };
    case PIN_GRAPH:
      return {
        ...state,
        generic: {
          ...state.generic,
          data: {
            ...state.generic.data,
            currentPageIdInHistory: action.payload,
          },
        },
      };
    case UNPIN_GRAPH:
      return {
        ...state,
        generic: {
          ...state.generic,
          data: {
            ...state.generic.data,
            currentPageIdInHistory: null,
          },
        },
      };
    case CLEAR_GRAPH:
      return DEFAULT_OLD_GRAPH_STATE;
    default:
      return state;
  }
};

export const oldGraphsReducer = (state = {}, action) => {
  if (!action.payload || !action.payload.graphId) {
    return state;
  }

  const { graphId, content } = action.payload;

  const oldGraphState = state[graphId] || DEFAULT_OLD_GRAPH_STATE;
  const oldGraphAction = { type: action.type, payload: content };

  return { ...state, [graphId]: oldGraphs(oldGraphState, oldGraphAction) };
};
