import * as uniqid from 'uniqid';
import UserLinksBasic from '../../../utils/UserLinksBasic';
import UrlUtils from '../../../utils/url/UrlUtils';
import * as API from '../../../api/oldRest';

/* Action creators */

const START_TO_LOAD_SELECTOR_VALUES = 'old/selectors/start_to_load';
const FINISH_TO_LOAD_SELECTOR_VALUES = 'old/selectors/finish_to_load';
const FAILED_TO_LOAD_SELECTOR_VALUES = 'old/selectors/failed_to_load';
const CLEAR_SELECTOR_VALUES = 'old/selectors/clear';

function convertResultToOptions(result, name, curUrl) {
  const options = [];

  const nameWithPrefix = UserLinksBasic.labelNameToQueryArgName(name);
  const urlParsed = UrlUtils.parseUrl(curUrl);
  const hasNameWithPrefix = urlParsed.hasArg(nameWithPrefix);

  for (let i = 0; i < result.values.length; ++i) {
    const value = result.values[i];

    let url;

    if (hasNameWithPrefix) {
      url = UrlUtils.updateParameter(curUrl, nameWithPrefix, value);
    } else {
      url = UrlUtils.updateParameter(
        UrlUtils.removeQueryArgs(curUrl, name),
        nameWithPrefix,
        value,
      );
    }

    options.push({ value, fullPath: url, selected: false });
  }

  return options;
}

export const getSelectorValues = (params, curUrl) => (dispatch) => {
  const { label } = params;

  const id = uniqid();

  dispatch({ type: START_TO_LOAD_SELECTOR_VALUES, payload: { id, label } });

  API.loadSelectorValues(params).then(
    (resp) => {
      const options = convertResultToOptions(resp, label, curUrl);
      return dispatch({
        type: FINISH_TO_LOAD_SELECTOR_VALUES,
        payload: { id, label, options },
      });
    },
    () => dispatch({ type: FAILED_TO_LOAD_SELECTOR_VALUES, payload: { id, label } }),
  );
};

export const clearSelectorValues = () => (dispatch) => dispatch({ type: CLEAR_SELECTOR_VALUES });

/* Reducers */

const DEFAULT_SELECTOR_STATE = {
  id: '',
  loading: false,
  data: null,
  error: null,
};

const DEFAULT_SELECTORS_STATE = {};

export const oldSelectorsReducer = (state = DEFAULT_SELECTORS_STATE, action) => {
  switch (action.type) {
    case START_TO_LOAD_SELECTOR_VALUES: {
      const { id, label } = action.payload;
      const oldSelector = state[label] || DEFAULT_SELECTOR_STATE;

      return {
        ...state,
        [label]: {
          ...oldSelector, id, loading: true, error: null,
        },
      };
    }
    case FINISH_TO_LOAD_SELECTOR_VALUES: {
      const { id, label } = action.payload;

      const oldSelector = state[label];

      if (oldSelector && oldSelector.id !== id) {
        return state;
      }

      return {
        ...state,
        [label]: {
          id, loading: false, data: action.payload.options, error: null,
        },
      };
    }
    case FAILED_TO_LOAD_SELECTOR_VALUES: {
      const { id, label } = action.payload;

      const oldSelector = state[label];

      if (oldSelector && oldSelector.id !== id) {
        return state;
      }

      return {
        ...state,
        [label]: {
          id, loading: false, data: null, error: 'Failed to load options',
        },
      };
    }
    case CLEAR_SELECTOR_VALUES:
      return DEFAULT_SELECTORS_STATE;
    default:
      return state;
  }
};
