import * as SensorsAPI from '../../../api/sensors';
import * as InternalAPI from '../../../api/internal';
import { addDangerAlert } from '../toaster';

const START_TO_LOAD_SENSORS_PAGE = 'old/START_TO_LOAD_SENSORS_PAGE';
const FINISH_TO_LOAD_SENSORS_PAGE = 'old/FINISH_TO_LOAD_SENSORS_PAGE';
const FAILED_TO_LOAD_SENSORS_PAGE = 'old/FAILED_TO_LOAD_SENSORS_PAGE';
const LOAD_FILTERED_LABELS = 'old/LOAD_FILTERED_LABELS';
const CLEAR_SENSORS_PAGE = 'old/CLEAR_SENSORS_PAGE';
const USER_PIN_DELETED_FROM_LIST = 'USER_PIN_DELETED_FROM_LIST';

/* Action creators */

export const getSensorsPage = (projectId, labelsParams, matchingParams, search) => (dispatch) => {
  dispatch({ type: START_TO_LOAD_SENSORS_PAGE });

  return SensorsAPI.fetchLabels(projectId, labelsParams).then(
    (labelsResp) => {
      const promises = [
        InternalAPI.getMatchingGraphs(projectId, matchingParams),
        InternalAPI.getMatchingDashboards(projectId, matchingParams),
        InternalAPI.getMatchingMenus(projectId, matchingParams),
        InternalAPI.getPinsByUrl({ url: `/${search}` }),
      ];

      return Promise.all(promises).then(
        (matchingResp) => dispatch({
          type: FINISH_TO_LOAD_SENSORS_PAGE,
          payload: {
            labels: labelsResp,
            matchingGraphs: matchingResp[0],
            matchingDashboards: matchingResp[1],
            matchingMenus: matchingResp[2],
            userPins: matchingResp[3],
          },
        }),
        () => dispatch({
          type: FINISH_TO_LOAD_SENSORS_PAGE,
          payload: {
            labels: labelsResp,
            matchingGraphs: [],
            matchingDashboards: [],
            matchingMenus: [],
          },
        }),
      );
    },
    (resp) => dispatch({ type: FAILED_TO_LOAD_SENSORS_PAGE, payload: resp.message }),
  );
};

export const getFilteredLabels = (projectId, params) => (dispatch) => {
  SensorsAPI.fetchLabels(projectId, params)
    .then((resp) => dispatch({ type: LOAD_FILTERED_LABELS, payload: resp }))
    .catch((resp) => dispatch(addDangerAlert(resp.message)));
};

export const deletePin = (id) => (dispatch) => InternalAPI.deletePin(id).then(
  () => dispatch({ type: USER_PIN_DELETED_FROM_LIST, payload: id }),
  (resp) => dispatch(addDangerAlert(resp.message)),
);

export const clearSensorsPage = () => (dispatch) => dispatch({ type: CLEAR_SENSORS_PAGE });

/* Reducer */

const DEFAULT_OLD_SENSORS_PAGE_STATE = {
  loading: false,
  data: null,
  error: null,
};

export const oldSensorsPageReducer = (state = DEFAULT_OLD_SENSORS_PAGE_STATE, action) => {
  switch (action.type) {
    case START_TO_LOAD_SENSORS_PAGE:
      return {
        ...state,
        loading: true,
        error: null,
      };
    case FINISH_TO_LOAD_SENSORS_PAGE:
      return {
        loading: false,
        data: action.payload,
      };
    case FAILED_TO_LOAD_SENSORS_PAGE:
      return {
        loading: false,
        error: action.payload,
      };
    case LOAD_FILTERED_LABELS:
      return {
        ...state,
        data: { ...state.data, labels: action.payload },
      };
    case USER_PIN_DELETED_FROM_LIST: {
      const id = action.payload;
      const newUserPins = state.data.userPins.filter((p) => p.id !== id);

      return {
        ...state,
        data: { ...state.data, userPins: newUserPins },
      };
    }
    case CLEAR_SENSORS_PAGE:
      return DEFAULT_OLD_SENSORS_PAGE_STATE;
    default:
      return state;
  }
};
