import * as API from '../../../api/projects';
import ProjectAuth from '../../../auth/ProjectAuth';
import { addDangerAlert } from '../toaster';

const START_TO_LOAD_PROJECT = 'START_TO_LOAD_PROJECT';
const FINISH_TO_LOAD_PROJECT = 'FINISH_TO_LOAD_PROJECT';
const FAILED_TO_LOAD_PROJECT = 'FAILED_TO_LOAD_PROJECT';
const PROJECT_SAVED = 'PROJECT_SAVED';
const PROJECT_DELETED = 'PROJECT_DELETED';
const PROJECT_CLEAR = 'PROJECT_CLEAR';

export const loadProject = (projectId) => (dispatch) => {
  dispatch({ type: START_TO_LOAD_PROJECT });

  return API.fetchProjectWithPermissions(projectId)
    .then((resp) => dispatch({
      type: FINISH_TO_LOAD_PROJECT,
      payload: resp,
    }),
    (resp) => dispatch({
      type: FAILED_TO_LOAD_PROJECT,
      payload: resp,
    }));
};

export const saveProject = (project, isNew) => (dispatch) =>
  API.saveProjectWithPermissions(project, isNew)
    .then((resp) => dispatch({
      type: PROJECT_SAVED,
      payload: resp,
    }),
    (resp) => dispatch(addDangerAlert(resp.message)));

export const deleteProject = (projectId) => (dispatch) => API.deleteProject(projectId)
  .then(
    () => dispatch({ type: PROJECT_DELETED }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );

export const clearProject = () => ({
  type: PROJECT_CLEAR,
});

const DEFAULT_STATE = {
  loading: false,
  project: {},
  projectAuth: null,
  error: null,
};

export const projectDataReducer = (state = DEFAULT_STATE, action) => {
  switch (action.type) {
    case START_TO_LOAD_PROJECT:
      return {
        loading: true,
        project: {},
        projectAuth: null,
        error: null,
      };
    case FINISH_TO_LOAD_PROJECT:
    case PROJECT_SAVED: {
      const { project, permissions } = action.payload;
      return {
        loading: false,
        project,
        projectAuth: ProjectAuth.from(project, permissions),
        error: null,
      };
    }
    case FAILED_TO_LOAD_PROJECT:
      return {
        loading: false,
        project: {},
        projectAuth: null,
        error: action.payload,
      };
    case PROJECT_CLEAR:
      return DEFAULT_STATE;
    default:
      return state;
  }
};
